// -*- Mode: C++; c-basic-offset: 4; indent-tabs-mode: nil; -*-
/**
 * Copyright 2006  Zack Rusin <zack@kde.org>
 * KDE 3 backport Copyright 2007 Kevin Kofler <Kevin@tigcc.ticalc.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
#include "enchantdict.h"
#include "enchantclient.h"

#include <qtextcodec.h>
#include <kdebug.h>

using namespace KSpell2;

QSpellEnchantDict::QSpellEnchantDict(QSpellEnchantClient *client, 
                                     EnchantBroker *broker,
                                     EnchantDict *dict,
                                     const QString &language)
    : Dictionary(language),
      m_broker(broker),
      m_dict(dict),
      m_client(client)
{
    kdDebug()<<"Enchant dict for"<<language << dict << endl;
}

QSpellEnchantDict::~QSpellEnchantDict()
{
    //Enchant caches dictionaries, so it will always return the same one.
    // therefore we do not want to delete the EnchantDict here but in the
    // client when it knows that nothing is using it anymore
    m_client->removeDictRef(m_dict);
}

bool QSpellEnchantDict::check(const QString &word)
{
    int wrong = enchant_dict_check(m_dict, word.utf8(),
                                   word.utf8().length());
    return !wrong;
}

QStringList QSpellEnchantDict::suggest(const QString &word)
{
    /* Needed for Unicode conversion */
    QTextCodec *codec = QTextCodec::codecForName("utf8");

    size_t number = 0;
    char **suggestions =
        enchant_dict_suggest(m_dict, word.utf8(), word.utf8().length(),
                             &number);

    QStringList qsug;
    for (size_t i = 0; i < number; ++i) {
        qsug.append(codec->toUnicode(suggestions[i]));
    }

    if (suggestions && number)
        enchant_dict_free_string_list(m_dict, suggestions);
    return qsug;
}

bool QSpellEnchantDict::checkAndSuggest(const QString& word,
                                        QStringList& suggestions)
{
    bool c = check(word);
    if (c)
        suggestions = suggest(word);
    return c;
}

bool QSpellEnchantDict::storeReplacement(const QString &bad,
                                  const QString &good)
{
    enchant_dict_store_replacement(m_dict,
                                   bad.utf8(), bad.utf8().length(),
                                   good.utf8(), good.utf8().length());
    return true;
}

bool QSpellEnchantDict::addToPersonal(const QString &word)
{
    kdDebug() << "QSpellEnchantDict::addToPersonal: word = "
              << word << endl;
    enchant_dict_add_to_pwl(m_dict, word.utf8(),
                            word.utf8().length());
    return true;
}

bool QSpellEnchantDict::addToSession(const QString &word)
{
    enchant_dict_add_to_session(m_dict, word.utf8(),
                                word.utf8().length());
    return true;
}
