/*
 * Protocol Buffers - Google's data interchange format
 * Copyright 2014 Google Inc.  All rights reserved.
 * https://developers.google.com/protocol-buffers/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.google.protobuf.jruby;

import org.jruby.*;
import org.jruby.anno.JRubyClass;
import org.jruby.anno.JRubyMethod;
import org.jruby.runtime.*;
import org.jruby.runtime.builtin.IRubyObject;

@JRubyClass(name = "Builder")
public class RubyBuilder extends RubyObject {
    public static void createRubyBuilder(Ruby runtime) {
        RubyModule internal = runtime.getClassFromPath("Google::Protobuf::Internal");
        RubyClass cBuilder = internal.defineClassUnder("Builder", runtime.getObject(), new ObjectAllocator() {
            @Override
            public IRubyObject allocate(Ruby runtime, RubyClass klazz) {
                return new RubyBuilder(runtime, klazz);
            }
        });
        cBuilder.defineAnnotatedMethods(RubyBuilder.class);
    }

    public RubyBuilder(Ruby runtime, RubyClass metaClass) {
        super(runtime, metaClass);
        this.cFileBuilderContext = (RubyClass) runtime.getClassFromPath("Google::Protobuf::Internal::FileBuilderContext");
    }

    /*
     * call-seq:
     *     Builder.new => builder
     *
     * Creates a new Builder. A Builder can accumulate a set of new message and enum
     * descriptors and atomically register them into a pool in a way that allows for
     * (co)recursive type references.
     */
    @JRubyMethod
    public IRubyObject initialize(ThreadContext context, IRubyObject descriptorPool) {
        this.descriptorPool = (RubyDescriptorPool) descriptorPool;
        return this;
    }

    /*
     * call-seq:
     *     Builder.add_message(name, &block)
     *
     * Old and deprecated way to create a new descriptor.
     * See FileBuilderContext.add_message for the recommended way.
     *
     * Exists for backwards compatibility to allow building descriptor pool for
     * files generated by protoc which don't add messages within "add_file" block.
     * Descriptors created this way get assigned to a default empty FileDescriptor.
     */
    @JRubyMethod(name = "add_message")
    public IRubyObject addMessage(ThreadContext context, IRubyObject name, Block block) {
        ensureDefaultFileBuilder(context);
        defaultFileBuilder.addMessage(context, name, block);
        return context.nil;
    }

    /*
     * call-seq:
     *     Builder.add_enum(name, &block)
     *
     * Old and deprecated way to create a new enum descriptor.
     * See FileBuilderContext.add_enum for the recommended way.
     *
     * Exists for backwards compatibility to allow building descriptor pool for
     * files generated by protoc which don't add enums within "add_file" block.
     * Enum descriptors created this way get assigned to a default empty
     * FileDescriptor.
     */
    @JRubyMethod(name = "add_enum")
    public IRubyObject addEnum(ThreadContext context, IRubyObject name, Block block) {
        ensureDefaultFileBuilder(context);
        defaultFileBuilder.addEnum(context, name, block);
        return context.nil;
    }

    /*
     * call-seq:
     *     Builder.add_file(name, options = nil, &block)
     *
     * Creates a new, file descriptor with the given name and options and invokes
     * the block in the context of a FileBuilderContext on that descriptor. The
     * block can then call FileBuilderContext#add_message or
     * FileBuilderContext#add_enum to define new messages or enums, respectively.
     *
     * This is the recommended, idiomatic way to build file descriptors.
     */
    @JRubyMethod(name = "add_file")
    public IRubyObject addFile(ThreadContext context, IRubyObject name, IRubyObject options, Block block) {
        RubyFileBuilderContext ctx = (RubyFileBuilderContext) cFileBuilderContext.newInstance(context, descriptorPool, name, options, Block.NULL_BLOCK);
        ctx.instance_eval(context, block);
        ctx.build(context);
        return context.nil;
    }

    /*
     * Used to trigger the build when using the deprecated syntax
     */
    protected void build(ThreadContext context) {
        if (defaultFileBuilder != null) {
            defaultFileBuilder.build(context);
        }
    }

    private void ensureDefaultFileBuilder(ThreadContext context) {
        if (defaultFileBuilder == null) {
            this.defaultFileBuilder = (RubyFileBuilderContext) cFileBuilderContext.newInstance(context, descriptorPool, context.runtime.newString("ruby_default_file.proto"), Block.NULL_BLOCK);
        }
    }

    private RubyClass cFileBuilderContext;
    private RubyDescriptorPool descriptorPool;
    private RubyFileBuilderContext defaultFileBuilder;
}
