
/*****************************************************************************
 *
 * Ethernetdriver for NE2000 compatible cards for ECOS
 * 
 * (C) 2001-mar-22 by Christian Plessl <cplessl@tik.ee.ethz.ch>
 *                    Thomas Meyer     <thomas.meyer@inalp.com>
 *
 *
 * ABSTRACT:
 *
 *  NE2000 Network Card Driver for eCos on an i386 compatible CPU.
 *
 *  Driver targets ISA network cards that are based on the Realtek 8019AS
 *  chipset which is used by most ne2000 clones. Make sure, that card is forced
 *  to jumper mode, since the driver doesnt support the ISA plug-n-play yet.
 *  IO-base address and IRQ used are hardwired in ne2000.h, change these to 
 *  your needs.
 *
 * HISTORY:
 *  Date       Initials   Description
 *  20001017   cp         created
 *  20000131   tm         reworked and summarized into one file
 *
 *****************************************************************************/

// -------------------------------------------
// The contents of this file are subject to the Red Hat eCos Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License.  You may obtain a copy of the License at
// http://www.redhat.com/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the
// License for the specific language governing rights and limitations under
// the License.
//
// The Original Code is eCos - Embedded Configurable Operating System,
// released September 30, 1998.
//
// The Initial Developer of the Original Code is Red Hat.
// Portions created by Red Hat are
// Copyright (C) 1998, 1999, 2000 Red Hat, Inc.
// All Rights Reserved.
// -------------------------------------------
//
// -------------------------------------------
//
// Portions of this software may have been derived from openblt
// http://www.openblt.org and are covered by the following 
// copyright disclaimer
//
// Brian J. Swetlands original copyright notice:
//
// Copyright 1998 Brian J. Swetland
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions, and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions, and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
///                                                                        

// -------------------------------------------                                 




// --- includes ------------------------------------------------------------

#include <pkgconf/system.h>
#include <cyg/infra/cyg_type.h>
#include <cyg/hal/hal_arch.h>
#include <cyg/infra/diag.h>
#include <eth_drv.h>
#include <netdev.h>
#include <cyg/hal/drv_api.h>

#include <pkgconf/net.h>

#include <cyg/kernel/kapi.h>
#include <stdlib.h>

#include "stdio.h"
#include "ne2000.h"


// --- defines -------------------------------------------------------------

// Hardware Configuration

#define NE2000_IOBASE 0x300
#define NE2000_IRQ        3



#define PROMISCUOUS (0)        // set to 1 to enable promiscouse mode

// Debugging Configuration

#undef  DEBUG_NE2000

// Calculated Hardware Configuration (don't touch)

#define NE2000_PC_IRQ (PC_HARDWARE_IRQ_OFFSET + (NE2000_IRQ))


// Debugging Macros
#ifndef DEBUG_NE2000
  #define dbg_diag_printf(format,args...) ;
#else
  #define dbg_diag_printf(format,args...) diag_printf(format,## args)
#endif


// --- Global variables ----------------------------------------------------

// FIXME:
// For the moment we have our data received in this global buffer
// in future we will use a fifo (linked list). so we can buffer multiple 
// frames.

struct buffer_header {
    unsigned char status;
    unsigned char next;
    unsigned short totallen; // length of header and packet 
};

struct nic_error_stat {
    long frame_alignment, crc, missed_packets;
    long rx, rx_size, rx_dropped, rx_fifo, rx_overruns;
    long tx_collisions;
    long tx_aborts, tx_carrier, tx_fifo, tx_heartbeat, tx_window;
};

struct nic_stat {
    long rx_packets;
    long tx_buffered, tx_packets;
    nic_error_stat errors;
};

struct packet_buffer {
    uint len;
    uint page;
    unsigned char *ptr;
    unsigned long key;
};


// FIXME:
// Remove when changing to fifo rx buffers.

struct if_ne2000_priv_data_t {
    bool ne2000_initialized; // true if card has been initialized

    int iobase;		       // NULL if uninitialized 
    int pstart, pstop, wordlength, current_page;
    nic_stat stat;
    packet_buffer tx_packet[MAX_TX], *last_tx;
    char busy, send;
  
    unsigned char prom[LEN_PROM];	// content of prom

    cyg_handle_t ne2000_int_handle;	// handle to inthandler
    cyg_interrupt ne2000_interrupt;	// the inthandler itself
    bool ne2000_int_initialized;	// is the int handler initialized
    bool ne2000_int_active;	        // is inthander active (attached and unmasked)
    unsigned char rx_dta[1600];
    unsigned int rx_len;
} priv_data;


ETH_DRV_SC (if_ne2000_sc, &priv_data,  // driver specific data
	    "eth0",
	    if_ne2000_start,
	    if_ne2000_stop,
	    if_ne2000_control,
	    if_ne2000_can_send,
	    if_ne2000_send,
	    if_ne2000_recv,
	    if_ne2000_deliver,
	    if_ne2000_poll,
	    if_ne2000_int_vector);

NETDEVTAB_ENTRY (if_ne2000_netdev,
		 "if_ne2000", if_ne2000_init, &if_ne2000_sc);


// --- forward declarations ------------------------------------------------

// ATTENTION: ECOS IO macros use another order of the arguments in the 
// outw and outb function
// ecos:  outb / outw (ioAddr, value)
// linux: outb / outw (value, ioAddr)

// i386 register access (providec by eCos)
extern int pc_inb (int port);
extern int pc_outb (int port, int value);
extern int pc_inw (int port);
extern int pc_outw (int port, int value);


void nic_isr (struct eth_drv_sc *sc);
void nic_tx  (struct eth_drv_sc *sc);
void nic_tx_err (struct eth_drv_sc *sc);
void nic_rx (struct eth_drv_sc *sc);
int nic_send (struct eth_drv_sc *sc, unsigned int buf);
int nic_dump_prom (struct eth_drv_sc *sc , unsigned char *prom);
int nic_init (struct eth_drv_sc *sc, int addr, unsigned char *prom, unsigned char *manual);
void nic_overrun (struct eth_drv_sc *sc);
void nic_block_input (struct eth_drv_sc *sc, unsigned char *buf, uint len, uint offset);
void nic_block_output (struct eth_drv_sc *sc, unsigned char *buf, uint len, uint page);


int if_ne2000_detect (int addr);

// -------------------------------------------------------------------------

// The handling of the nic interrupt is left to the DSR if_ne2000_deliver
cyg_uint32 if_ne2000_isr (cyg_vector_t vector, cyg_addrword_t data)
{
    cyg_drv_interrupt_mask (NE2000_PC_IRQ);	// mask nic interrupt
    return (CYG_ISR_HANDLED | CYG_ISR_CALL_DSR);	// run DSR
}

// -------------------------------------------------------------------------

// This function runs as dsr for the nic interrupt.
void
if_ne2000_deliver (struct eth_drv_sc *sc)
{
    nic_isr (sc);	                        // call interrupt handler
    cyg_interrupt_acknowledge (NE2000_PC_IRQ);	// ack interrupt (reset isr)
    cyg_interrupt_unmask (NE2000_PC_IRQ);	// reenable interrupt
}

// -------------------------------------------------------------------------

static bool
if_ne2000_init (struct cyg_netdevtab_entry *tab)
{
    struct eth_drv_sc *sc = (struct eth_drv_sc *) tab->device_instance;
    struct if_ne2000_priv_data_t *dp =
	(struct if_ne2000_priv_data_t *) sc->driver_private;
    unsigned char prom[16];
    unsigned int f;

    if (!if_ne2000_detect (NE2000_IOBASE) == NE2000_IOBASE) {
	dbg_diag_printf ("ne2000: not card found at 0x%X\n", NE2000_IOBASE);
	return false;
    }

    dp->iobase = NE2000_IOBASE;
    nic_stat_clear (&(dp->stat));
    dp->pstart = 0;
    dp->pstop = 0;
    dp->wordlength = 0;
    dp->current_page = 0;
    for (f = 0; f < MAX_TX; f++) {
        dp->tx_packet[f].len = 0;
    }
    dp->last_tx = NULL;
    dp->busy = 0;
    
    pc_outb (dp->iobase + NE_RESET, pc_inb (dp->iobase + NE_RESET));	// reset NE2000 

    while (!(pc_inb (dp->iobase + INTERRUPTSTATUS) & ISR_RST)) {
        // FIXME:  insert timeout here. 
    }

    pc_outb (dp->iobase + INTERRUPTSTATUS, 0xff);	// clear all pending ints 

    // Initialize registers
    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_STOP);	// enter page 0 
    pc_outb (dp->iobase + DATACONFIGURATION, DCR_DEFAULT);
    pc_outb (dp->iobase + REMOTEBYTECOUNT0, 0x00);
    pc_outb (dp->iobase + REMOTEBYTECOUNT1, 0x00);
    pc_outb (dp->iobase + INTERRUPTMASK, 0x00);	// mask all ints 
    pc_outb (dp->iobase + INTERRUPTSTATUS, 0xff);	// clear pending ints 
    pc_outb (RECEIVECONFIGURATION, RCR_MON);	// enter monitor mode 
    pc_outb (TRANSMITCONFIGURATION, TCR_INTERNAL_LOOPBACK);	// internal loopback 

    dp->wordlength = nic_dump_prom (sc, &(dp->prom));
    if (dp->prom[14] != 0x57 || dp->prom[15] != 0x57) {
        dbg_diag_printf("ne2000: cannot initialize card, something wrong with eeprom\n");
	return false;
    }

    // set wordlenght of data port
    if (dp->wordlength == 2) {
	pc_outb (dp->iobase + DATACONFIGURATION, DCR_DEFAULT_WORD);
    }
    dp->pstart = (dp->wordlength == 2) ? PSTARTW : PSTART;
    dp->pstop = (dp->wordlength == 2) ? PSTOPW : PSTOP;

    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_STOP);
    pc_outb (dp->iobase + TRANSMITPAGE, dp->pstart);	// setup local buffer 
    pc_outb (dp->iobase + PAGESTART, dp->pstart + TXPAGES);
    pc_outb (dp->iobase + BOUNDARY, dp->pstop - 1);
    pc_outb (dp->iobase + PAGESTOP, dp->pstop);
    pc_outb (dp->iobase + INTERRUPTMASK, 0x00);	// mask all ints 
    pc_outb (dp->iobase + INTERRUPTSTATUS, 0xff);	// clear pending ints 

    dp->current_page = dp->pstart + TXPAGES;

    // Setup MAC address
    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE1 | NIC_STOP);	// switch to page 1 
    for (f = 0; f < LEN_ADDR; f++) {
        pc_outb (dp->iobase + PHYSICAL + f, dp->prom[f]);
    }
        
    // Setup multicast filter to accept all multicast packets
    for (f = 0; f < 8; f++) {
	pc_outb (dp->iobase + MULTICAST + f, 0xff);
    }

    pc_outb (dp->iobase + CURRENT, dp->pstart + TXPAGES);
    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_STOP);	//  switch back to page 0 

    dp->ne2000_initialized = true; // card is now initialized
    dp->rx_len = 0;
	
    // Callback: Notify generic ethernet driver about initialization
    (sc->funs->eth_drv->init) (sc, &(dp->prom[0]));

    // init int handler
    if (!dp->ne2000_int_initialized) {
        cyg_drv_interrupt_create (NE2000_PC_IRQ,
                                  15,  // priority ?
                                  (cyg_addrword_t) sc,
                                  if_ne2000_isr,
                                  (cyg_DSR_t *) eth_drv_dsr,
                                  &(dp->ne2000_int_handle),
                                  &(dp->ne2000_interrupt));
        dp->ne2000_int_initialized = true;
        dp->ne2000_int_active = false;
        dbg_diag_printf ("ne2000: interrupt created\n");
    }
    else {
        dbg_diag_printf ("ne2000: interrupt was already created\n");
    }
    dbg_diag_printf
        ("ne2000: card at 0x%x atsuccessfully initialized\n",
         NE2000_IOBASE);
    return true;
}


// -------------------------------------------------------------------------

/* Can be called serveral times, even if already running. We should
 * handle this case in future.
 */

static void
if_ne2000_start (struct eth_drv_sc *sc, unsigned char *enaddr, int flags)
{
    struct if_ne2000_priv_data_t *dp =
	(struct if_ne2000_priv_data_t *) sc->driver_private;

    int promiscuous = PROMISCUOUS;

    if (!dp->iobase) {
	return;
    }

    pc_outb (dp->iobase + INTERRUPTSTATUS, 0xff);
    pc_outb (dp->iobase + INTERRUPTMASK, IMR_DEFAULT);
    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_START);
    pc_outb (dp->iobase + TRANSMITCONFIGURATION, TCR_DEFAULT);
    if (promiscuous) {
	pc_outb (dp->iobase + RECEIVECONFIGURATION, RCR_PRO | RCR_AM);
    }
    else {
	pc_outb (dp->iobase + RECEIVECONFIGURATION, RCR_DEFAULT);
    }

    dbg_diag_printf ("ne2000: started interface %s\n", sc->dev_name);

    if (!dp->ne2000_int_active) {
	cyg_interrupt_attach (dp->ne2000_int_handle);
	cyg_drv_interrupt_acknowledge (NE2000_PC_IRQ);
	cyg_drv_interrupt_unmask (NE2000_PC_IRQ);

	dp->ne2000_int_active = true;
	dbg_diag_printf ("ne2000: inthandler attached, int unmasked\n");
    }
}

// -------------------------------------------------------------------------

static void
if_ne2000_stop (struct eth_drv_sc *sc)
{
    struct if_ne2000_priv_data_t *dp =
	(struct if_ne2000_priv_data_t *) sc->driver_private;
    unsigned char tmp_buffer[16];

    if (!dp->iobase) {
        return;			       // make sure card was initialized 
    }

#if 0
    // why should we re-initialize the nic immediately ?
    nic_init (sc, dp->iobase, tmp_buffer, NULL);  
#endif

    dbg_diag_printf ("ne2000: stopped interface %s\n", sc->dev_name);
    
    if ((dp->ne2000_int_initialized) && !(dp->ne2000_int_active)) {
	// Disable int handler
	cyg_drv_interrupt_mask (NE2000_PC_IRQ);
	cyg_drv_interrupt_detach (dp->ne2000_int_handle);
	cyg_drv_interrupt_acknowledge (NE2000_PC_IRQ);
	dbg_diag_printf ("ne2000: deactivated int handler\n");
    }
    else {
	dbg_diag_printf ("ne2000: int handler already deactivated\n");
    }
}

// -------------------------------------------------------------------------

static int
if_ne2000_control (struct eth_drv_sc *sc, unsigned long key,
		   void *data, int len)
{
    // FIXME:
    // We just ignore this for the moment

    switch (key) {
    case ETH_DRV_SET_MAC_ADDRESS:
        return 0;
        break;

    default:
        return 1;
        break;
    }
}

// -------------------------------------------------------------------------

// For this version of this driver, we allow queuing a new packet only
// if the transmit buffer is empty at the moment, e.g. device is not busy..

static int
if_ne2000_can_send (struct eth_drv_sc *sc)
{
    struct if_ne2000_priv_data_t *dp =
	(struct if_ne2000_priv_data_t *) sc->driver_private;

    if (dp->busy) {
	dbg_diag_printf ("ne2000: transmitting: queuing of new packet "
			 "not allowed\n");
	return 0;
    }
    else {
	dbg_diag_printf
	    ("ne2000: transmitter idle: queuing of new packet allowed\n");
	return 1;
    }
}

// -------------------------------------------------------------------------

static void
if_ne2000_send (struct eth_drv_sc *sc, struct eth_drv_sg *sg_list,
		int sg_len, int total_len, unsigned long key)
{
    struct if_ne2000_priv_data_t *dp =
	(struct if_ne2000_priv_data_t *) sc->driver_private;

    static packet_buffer buf;
    static unsigned char pktdata[1500];	// replace by MAX_LENGTH constant
    int i, k, bufptr, len;
    unsigned char *dataptr;
    unsigned int f;

    // FIXME evl noch checks, ob nic bereits richtig initialisiert wurde, 
    // d.h. nic->iobase gesetzt.

    // evl. errorchecking ob len> MAX_LENGTH

    for (i = 0; i < sizeof (pktdata); i++) {
	pktdata[i] = '\x00';	       // zero out buffer
    }
    buf.len = 0;
    buf.key = key;
    buf.ptr = pktdata;
    bufptr = 0;

    for (i = 0; i < sg_len; i++) {     // iterate over all buffers
	dataptr = (unsigned char *) sg_list[i].buf;
	len = sg_list[i].len;
	buf.len += len;

	for (k = 0; k < len; k++) {
	    pktdata[bufptr++] = *dataptr++;
	}
    }

    if (total_len != buf.len) {
	dbg_diag_printf ("if_ne2000_send: len missmatch! total_length != "
			 "sum of legth of individual buffers\n");
    }
    
    if (dp->busy) {
      dbg_diag_printf ("if_ne2000_send: tried sending new packet while busy\n");
      return;
    }

    dp->busy = 1; //mark nic busy

    if (buf.len < MIN_LENGTH){
        buf.len = MIN_LENGTH;
    }

    pc_outb (dp->iobase + INTERRUPTMASK, 0x00);	// mask ints

    // iterate over all transmit buffers
    for (f = 0; f < MAX_TX; f++) {
        if (dp->tx_packet[f].len == 0) {
            dp->tx_packet[f] = buf;
	    dp->tx_packet[f].page = dp->pstart + (f * MAX_PAGESPERPACKET);

	    // Output page
	    dp->tx_packet[f].key = buf.key;
	    nic_block_output (sc, dp->tx_packet[f].ptr,
			      dp->tx_packet[f].len, dp->tx_packet[f].page);
	    dp->send = f;
            // Now let's actually trigger the transmitter to send
	    if (nic_send (sc, f) < 0) {
		break;
	    }

	    // Note: the nic_tx() interrupt will mark this tx_packet buffer as
	    // available again once it confirms that the packet was sent.
	    dp->stat.tx_buffered++;
	    pc_outb (dp->iobase + INTERRUPTMASK, IMR_DEFAULT);	// unmask ints again
            dp->busy = 0;
	    return; // buffer sent
	}
    }

    // could not send buffer, so unmask ints again and return
    pc_outb (dp->iobase + INTERRUPTMASK, IMR_DEFAULT);
    dp->busy = 0;
    return;  // error
}

// -------------------------------------------------------------------------

static void
if_ne2000_recv (struct eth_drv_sc *sc, struct eth_drv_sg *sg_list, int sg_len)
{
    unsigned int i;
    unsigned int total_len = 0;
    unsigned int thisbuflen = 0;
    struct if_ne2000_priv_data_t *dp =
	(struct if_ne2000_priv_data_t *) sc->driver_private;
    unsigned char *dataptr;
    unsigned char *rx_dataptr;

    rx_dataptr = dp->rx_dta;

    for (i = 0; i < sg_len; i++)
	total_len += sg_list[i].len;
    dbg_diag_printf ("ne2000: ethernet layer callback: if_ne2000_recv\n");
    dbg_diag_printf ("      enet wants %d bytes of data to %d buffers\n",
		     total_len, sg_len);

    dbg_diag_printf
	("ne2000: we have currently a packet of size %d in buffer\n",
	 dp->rx_len);
    if (total_len != dp->rx_len) {
	dbg_diag_printf ("      ## size mismatch, cropping packet to len "
			 "of sg_buffers\n");
    }

    for (i = 0; i < sg_len; i++) {
	dataptr = (unsigned char *) sg_list[i].buf;
	if (dataptr) {
	    thisbuflen = sg_list[i].len;
	    while (thisbuflen--) {
		*dataptr++ = *rx_dataptr++;
	    }
	}
    }
    dbg_diag_printf
	("ne2000: delivered 1 ethernet frame to ethernet layer\n");

    dp->rx_len = 0;		       // mark buffer as available again!

}

// -------------------------------------------------------------------------

static void
if_ne2000_poll (struct eth_drv_sc *sc)
{
    nic_isr (sc);
}

// -------------------------------------------------------------------------

static int
if_ne2000_int_vector (struct eth_drv_sc *sc)
{
    return NE2000_PC_IRQ;
}

// -------------------------------------------------------------------------

// Check if NE2000 card is present at address addr. 
// Return addr if card was detected, otherwise return -1.
int
if_ne2000_detect (int addr)
{
    unsigned int regd;
    unsigned int state;

    state = pc_inb (addr);	// backup state of addr

    pc_outb (addr, NIC_DMA_DISABLE | NIC_PAGE1 | NIC_STOP);
    regd = pc_inb (addr + 0x0d);
    pc_outb (addr + 0x0d, 0xff);
    pc_outb (addr, NIC_DMA_DISABLE | NIC_PAGE0);
    pc_inb (addr + FAE_TALLY);	       // reading CNTR0 resets it. 

    if (pc_inb (addr + FAE_TALLY)) {   // counter didn't clear so probe fails 
	pc_outb (addr, state);	       // try to restore state 
	pc_outb (addr + 0x0d, regd);
	return -1;
    }
    return addr; // network card detected at io addr; 
}

// -------------------------------------------------------------------------

// If manual set != null card is MAC address is set to address pointed to 
// by manual.
// Dumps the content of the prom into buffer prom.

int
nic_init (struct eth_drv_sc *sc, int addr, unsigned char *prom, unsigned char *manual)
{
    unsigned int f;
    struct if_ne2000_priv_data_t *dp =
        (struct if_ne2000_priv_data_t *) sc->driver_private;

    if (!dp->iobase) {
	dp->iobase = addr;
	nic_stat_clear (&(dp->stat));
	dp->pstart = 0;
	dp->pstop = 0;
	dp->wordlength = 0;
	dp->current_page = 0;
	for (f = 0; f < MAX_TX; f++) {
	    dp->tx_packet[f].len = 0;
	}
	dp->last_tx = NULL;
	dp->busy = 0;
    }
    else {
	if (!dp->iobase || dp->iobase != addr) {
	    return -1;
	}
    }

    pc_outb (addr + NE_RESET, pc_inb (addr + NE_RESET));	// reset the NE2000 

    while (!(pc_inb (addr + INTERRUPTSTATUS) & ISR_RST)) {
    // TODO insert timeout code here. 
    }

    pc_outb (addr + INTERRUPTSTATUS, 0xff);	// clear all pending ints 

    // Initialize registers

    pc_outb (addr, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_STOP);	// enter page 0 
    pc_outb (addr + DATACONFIGURATION, DCR_DEFAULT);
    pc_outb (addr + REMOTEBYTECOUNT0, 0x00);
    pc_outb (addr + REMOTEBYTECOUNT1, 0x00);
    pc_outb (addr + INTERRUPTMASK, 0x00);	// mask all ints 
    pc_outb (addr + INTERRUPTSTATUS, 0xff);	// clear pending ints 
    pc_outb (RECEIVECONFIGURATION, RCR_MON);	// enter monitor mode 
    pc_outb (TRANSMITCONFIGURATION, TCR_INTERNAL_LOOPBACK);	// internal loopback 

    dp->wordlength = nic_dump_prom (sc, prom);
    if (prom[14] != 0x57 || prom[15] != 0x57) {
	return -1;
    }

    // set wordlenght of data port
    if (dp->wordlength == 2) {
	pc_outb (addr + DATACONFIGURATION, DCR_DEFAULT_WORD);
    }
    dp->pstart = (dp->wordlength == 2) ? PSTARTW : PSTART;
    dp->pstop = (dp->wordlength == 2) ? PSTOPW : PSTOP;

    pc_outb (addr, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_STOP);
    pc_outb (addr + TRANSMITPAGE, dp->pstart);	// setup local buffer 
    pc_outb (addr + PAGESTART, dp->pstart + TXPAGES);
    pc_outb (addr + BOUNDARY, dp->pstop - 1);
    pc_outb (addr + PAGESTOP, dp->pstop);
    pc_outb (addr + INTERRUPTMASK, 0x00);	// mask all ints 
    pc_outb (addr + INTERRUPTSTATUS, 0xff);	// clear pending ints 

    dp->current_page = dp->pstart + TXPAGES;

    // Put physical address in the registers
    pc_outb (addr, NIC_DMA_DISABLE | NIC_PAGE1 | NIC_STOP);	// switch to page 1 

    if (manual) {
	for (f = 0; f < 6; f++) {
	    pc_outb (addr + PHYSICAL + f, manual[f]);
	}
    }
    else {
	for (f = 0; f < LEN_ADDR; f++) {
	    pc_outb (addr + PHYSICAL + f, prom[f]);
	}
    }

    // Setup multicast filter to accept all packets
    for (f = 0; f < 8; f++) {
	pc_outb (addr + MULTICAST + f, 0xff);
    }

    pc_outb (addr + CURRENT, dp->pstart + TXPAGES);
    pc_outb (addr, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_STOP);	//  switch back to page 0 

    return 0;
}

// -------------------------------------------------------------------------

void
nic_isr (struct eth_drv_sc *sc)
{
    unsigned int isr;
    unsigned int overload;

    struct if_ne2000_priv_data_t *dp =
      (struct if_ne2000_priv_data_t *) sc->driver_private;

    if (!dp->iobase) {  // check if card was initialized
	return;
    }
    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0);
    overload = MAX_LOAD + 1;
    while ((isr = pc_inb (dp->iobase + INTERRUPTSTATUS))) {
	if ((--overload) <= 0) {
	    break;
	}
	if (isr & ISR_OVW) {
	    nic_overrun (sc);
	}
	else if (isr & (ISR_PRX | ISR_RXE)) {
	    nic_rx (sc);
	}

	if (isr & ISR_PTX) {
	    nic_tx (sc);
	}
	else if (isr & ISR_TXE) {
	    nic_tx_err (sc);
	}

	if (isr & ISR_RDC) {
	    pc_outb (dp->iobase + INTERRUPTSTATUS, ISR_RDC);
	}

	pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_START);
    }

    if (isr) {
	pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_START);
	if (!overload) {
	    pc_outb (dp->iobase + INTERRUPTSTATUS, ISR_ALL);	// clear
	}
	else {
	    pc_outb (dp->iobase + INTERRUPTSTATUS, 0xff);
	    // Ack it anyway
	}
    }
}

// -------------------------------------------------------------------------

/*
 * You should call this before you just read the stats directlly from the
 * snic struct.  This procedure updates the counters 
 */

nic_stat nic_get_stats (struct eth_drv_sc *sc)
{
    struct if_ne2000_priv_data_t *dp =
        (struct if_ne2000_priv_data_t *) sc->driver_private;
    dp->stat.errors.frame_alignment += pc_inb (dp->iobase + FAE_TALLY);
    dp->stat.errors.crc += pc_inb (dp->iobase + CRC_TALLY);
    dp->stat.errors.missed_packets += pc_inb (dp->iobase + MISS_PKT_TALLY);
    return dp->stat;
}

// -------------------------------------------------------------------------

void
nic_stat_clear (nic_stat * that)
{
    that->rx_packets = 0;
    that->tx_buffered = 0;
    that->tx_packets = 0;
    that->errors.frame_alignment = 0;
    that->errors.crc = 0;
    that->errors.missed_packets = 0;
    that->errors.rx = 0;
    that->errors.rx_size = 0;
    that->errors.rx_dropped = 0;
    that->errors.rx_fifo = 0;
    that->errors.rx_overruns = 0;
    that->errors.tx_collisions = 0;
}

// -------------------------------------------------------------------------

/*
 * Dumps the prom into a 16 byte buffer and returns the wordlength of the
 * card. You should be able to make this procedure a wrapper of
 * nic_block_input().
 */

int
nic_dump_prom (struct eth_drv_sc *sc, unsigned char *prom)
{
    unsigned int f;

    struct if_ne2000_priv_data_t *dp =
        (struct if_ne2000_priv_data_t *) sc->driver_private;
   
    char wordlength = 2; // default wordlength is 2
    unsigned char dump[32];

    pc_outb (dp->iobase + REMOTEBYTECOUNT0, 32);	// read 32 bytes from DMA->IO 
    pc_outb (dp->iobase + REMOTEBYTECOUNT1, 0x00);	// this is for the PROM dump 
    pc_outb (dp->iobase + REMOTESTARTADDRESS0, 0x00);	// configure DMA for 0x0000 
    pc_outb (dp->iobase + REMOTESTARTADDRESS1, 0x00);
    pc_outb (dp->iobase, NIC_REM_READ | NIC_START);
    for (f = 0; f < 32; f += 2) {
	dump[f] = pc_inb (dp->iobase + NE_DATA);
	dump[f + 1] = pc_inb (dp->iobase + NE_DATA);
	if (dump[f] != dump[f + 1])
	    wordlength = 1;
    }

    //If wordlength is 2 bytes, then collapse prom to 16 bytes
    for (f = 0; f < LEN_PROM; f++)
	prom[f] = dump[f + ((wordlength == 2) ? f : 0)];

     return wordlength;
}

// -------------------------------------------------------------------------

void
nic_overrun (struct eth_drv_sc *sc)
{
    unsigned int tx_status;

    struct if_ne2000_priv_data_t *dp =
        (struct if_ne2000_priv_data_t *) sc->driver_private;

    unsigned int resend = 0;

    if (!dp->iobase)
	return;
    tx_status = pc_inb (dp->iobase) & NIC_TRANSMIT;
    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_STOP);
    dp->stat.errors.rx_overruns++;

    /*
     * FIXME:
     * Replace this whole crappy code with a decent wait
     * method.  We need to wait at least 1.6ms as per National Semiconductor 
     * datasheets, but we should probablly wait a litle more to be safe.
     */

    pc_outb (dp->iobase + REMOTEBYTECOUNT0, 0x00);
    pc_outb (dp->iobase + REMOTEBYTECOUNT1, 0x00);
    if (tx_status) {
	unsigned int tx_completed =
	    pc_inb (dp->iobase + INTERRUPTSTATUS) & (ISR_PTX | ISR_TXE);

	if (!tx_completed) {
	    resend = 1;
	}
    }

    pc_outb (dp->iobase + TRANSMITCONFIGURATION, TCR_INTERNAL_LOOPBACK);
    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_START);
    nic_rx (sc);		       // cleanup RX ring 
    pc_outb (dp->iobase + INTERRUPTSTATUS, ISR_OVW);	// ACK INT 
    pc_outb (dp->iobase + TRANSMITCONFIGURATION, TCR_DEFAULT);
    if (resend) {
	pc_outb (dp->iobase,
		 NIC_DMA_DISABLE | NIC_PAGE0 | NIC_START | NIC_TRANSMIT);
    }
}

// -------------------------------------------------------------------------

/*
 * This is the procedure that marks the transmit buffer as available again
 */

void
nic_tx (struct eth_drv_sc *sc)
{
    unsigned int f;

    struct if_ne2000_priv_data_t *dp = (struct if_ne2000_priv_data_t *) sc->driver_private;
    unsigned int status;

    if (!dp->iobase) {
	return;
    }
    status = pc_inb (dp->iobase + TRANSMITSTATUS);

    if (!dp->tx_packet[dp->send].len) {
	return;
    }
    
    // notify ethernet layer that packet has been sent.
    (sc->funs->eth_drv->tx_done) (sc,
                                  dp->tx_packet[dp->send].key,
                                  1);

    dp->tx_packet[dp->send].len = 0; // mark buffer as available again

    for (f = 0; f < MAX_TX; f++) {
	if (dp->tx_packet[f].len) {
	    dp->stat.tx_buffered++;
	    nic_send (sc, f);	       // send a back-to-back buffer 
	    break;
	}
    }

    if (status & TSR_COL) {
	dp->stat.errors.tx_collisions++;
    }

    if (status & TSR_PTX) {
	dp->stat.tx_packets++;
    }
    else {
	if (status & TSR_ABT) {
	    dp->stat.errors.tx_aborts++;
	    dp->stat.errors.tx_collisions += 16;
	}
	if (status & TSR_CRS) {
	    dp->stat.errors.tx_carrier++;
	}
	if (status & TSR_FU) {
	    dp->stat.errors.tx_fifo++;
	}
	if (status & TSR_CDH) {
	    dp->stat.errors.tx_heartbeat++;
	}
	if (status & TSR_OWC) {
	    dp->stat.errors.tx_window++;
	}
    }
    pc_outb (dp->iobase + INTERRUPTSTATUS, ISR_PTX); // ack interrupts
}

// -------------------------------------------------------------------------

// FIXME: removed printing of error messages, since this function is called
// from dsr routine. How should we implement a clean error handling?
void
nic_tx_err (struct eth_drv_sc *sc)
{
    unsigned char tsr;

    struct if_ne2000_priv_data_t *dp =
        (struct if_ne2000_priv_data_t *) sc->driver_private;

    if (!dp->iobase) {
	return;
    }
    tsr = pc_inb (dp->iobase);

    // TX error
    if (tsr & TSR_ABT) {
        // Too many collisions
    }
    if (tsr & TSR_ND) {
	// Not defered
    }
    if (tsr & TSR_CRS) {
        // Carrier lost
    }
    if (tsr & TSR_FU) {
       // FIFO underrun
    }
    if (tsr & TSR_CDH) {
        // Heartbeat lost
    }

    pc_outb (dp->iobase + INTERRUPTSTATUS, ISR_TXE);
    if (tsr & (TSR_ABT | TSR_FU)) {
	nic_tx (sc);
    }
}

// -------------------------------------------------------------------------

void
nic_rx (struct eth_drv_sc *sc)
{
    unsigned int packets = 0;
    unsigned int frame, rx_page, rx_offset, numpages;
    unsigned int len, next_pkt;

    struct if_ne2000_priv_data_t *dp =
        (struct if_ne2000_priv_data_t *) sc->driver_private;
    buffer_header header;
    int i;

    if (!dp->iobase) {
	return;
    }

    while (packets < MAX_RX) {
	pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE1);	// curr is on page 1 
	rx_page = pc_inb (dp->iobase + CURRENT);	// get current page 
	pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0);
	frame = pc_inb (dp->iobase + BOUNDARY) + 1;

	// We add one because boundary is a page behind as pre NS notes 
        // to help in overflow problems

	if (frame >= dp->pstop) {
	    frame = dp->pstart + TXPAGES;	// circual buffer 
	}

	if (frame == rx_page) {
	    break;	// all frames read 
	}

	rx_offset = frame << 8;	       // current ptr in bytes(not pages) 

	nic_block_input (sc, (unsigned char *) &header,
			 sizeof (buffer_header), rx_offset);

	len = header.totallen - sizeof (buffer_header);	// length of packet 
	next_pkt = frame + 1 + ((len + 4) >> 8);	// next packet frame 

	numpages = dp->pstop - (dp->pstart + TXPAGES);

        //Index mismatch
	if ((header.next != next_pkt)
	    && (header.next != next_pkt + 1)
	    && (header.next != next_pkt - numpages)
	    && (header.next != next_pkt + 1 - numpages)) {
	    dp->current_page = frame;
	    pc_outb (dp->iobase + BOUNDARY, dp->current_page - 1);
	    dp->stat.errors.rx++;
	    continue;
	}

	if (len < 60 || len > 1518) {
            // invalid packet size
	    dp->stat.errors.rx_size++;
	}
	else if ((header.status & 0x0f) == RSR_PRX) {
           // We have a good packet, so let's recieve it! 
	   
	    if (dp->rx_len != 0) {
		// losing packet, since we have just one single buffer, which 
                // is already used at the moment.
		dp->stat.errors.rx_dropped++;
		break;
	    }

	    for (i = 0; i < sizeof (dp->rx_dta); i++) {
		dp->rx_dta[i] = '\0';
	    }

	    // Length packet might not be the same as length of payload. If payload 
	    // length is less than 60bytes, ethernet frames are padded to 60 bytes. 

	    dp->rx_len = len;	       
            //  only if payload > 60bytes, so we  adjust this in the following

	    // read packet to global buffer
	    nic_block_input (sc, dp->rx_dta, dp->rx_len,
			     rx_offset + sizeof (buffer_header));

	    // announce reception of new packet to ethernet layer.
	    (sc->funs->eth_drv->recv) (sc, len);
	    dp->stat.rx_packets++;
	}
	else {
            // bad packet header
	    if (header.status & RSR_FO) {
		dp->stat.errors.rx_fifo++;
	    }
	}
	next_pkt = header.next;

	if (next_pkt >= dp->pstop) {
            // next frame beyond local buffer!"
	    next_pkt = dp->pstart + TXPAGES;
	}

	dp->current_page = next_pkt;
	pc_outb (dp->iobase + BOUNDARY, next_pkt - 1);
    }
    pc_outb (dp->iobase + INTERRUPTSTATUS, ISR_PRX | ISR_RXE);	// ack int 
}

// -------------------------------------------------------------------------
int
nic_send (struct eth_drv_sc *sc, unsigned int buf)
{

    struct if_ne2000_priv_data_t *dp =
        (struct if_ne2000_priv_data_t *) sc->driver_private;

    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0);
    if (pc_inb (dp->iobase + STATUS) & NIC_TRANSMIT) {
        // cannot send, transmitter busy
	dp->tx_packet[buf].len = 0;   // mark as free again 
	return -1;
    }
    pc_outb (dp->iobase + TRANSMITBYTECOUNT0,
	     dp->tx_packet[buf].len & 0xff);
    pc_outb (dp->iobase + TRANSMITBYTECOUNT1, dp->tx_packet[buf].len >> 8);
    pc_outb (dp->iobase + TRANSMITPAGE, dp->tx_packet[buf].page);
    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_TRANSMIT | NIC_START);
    return 0;
}

// -------------------------------------------------------------------------

// reading data from ne2000 to buffer buf with lenght len starting at offset offset.
void
nic_block_input (struct eth_drv_sc *sc, unsigned char *buf, unsigned int len,
		 unsigned int offset)
{
    struct if_ne2000_priv_data_t *dp =
        (struct if_ne2000_priv_data_t *) sc->driver_private;

    unsigned int f;
    unsigned int xfers = len;
    unsigned int timeout = TIMEOUT_DMAMATCH;
    unsigned int addr;

    pc_outb (dp->iobase, NIC_DMA_DISABLE | NIC_PAGE0 | NIC_START);
    pc_outb (dp->iobase + REMOTEBYTECOUNT0, len & 0xff);
    pc_outb (dp->iobase + REMOTEBYTECOUNT1, len >> 8);
    pc_outb (dp->iobase + REMOTESTARTADDRESS0, offset & 0xff);
    pc_outb (dp->iobase + REMOTESTARTADDRESS1, offset >> 8);
    pc_outb (dp->iobase, NIC_REM_READ | NIC_START);

    if (dp->wordlength == 2) {
	for (f = 0; f < (len >> 1); f++) {
	    ((unsigned short *) buf)[f] = pc_inw (dp->iobase + NE_DATA);
	}
	if (len & 0x01) {
	    ((unsigned char *) buf)[len - 1] = pc_inb (dp->iobase + NE_DATA);
	    xfers++;
	}
    }
    else {
	for (f = 0; f < len; f++) {
	    ((unsigned char *) buf)[f] = pc_inb (dp->iobase + NE_DATA);
	}
    }

    // FIXME:
    // Make this timeout a constant
    
    for (f = 0; f < timeout; f++) {
	unsigned int high = pc_inb (dp->iobase + REMOTESTARTADDRESS1);
	unsigned int low = pc_inb (dp->iobase + REMOTESTARTADDRESS0);

	addr = (high << 8) + low;
	if (((offset + xfers) & 0xff) == low) {
	    break;
	}
    }
    if (f >= timeout) {
        // remote DMA address invalid expected offset + xfers, actual is addr
    }

    pc_outb (dp->iobase + INTERRUPTSTATUS, ISR_RDC);      // finish DMA cycle 
}

// -------------------------------------------------------------------------

// What is this timeout good for?
 
void
nic_block_output (struct eth_drv_sc *sc, unsigned char *buf, unsigned int len,
		  unsigned int page)
{
    struct if_ne2000_priv_data_t *dp =
        (struct if_ne2000_priv_data_t *) sc->driver_private;


    int timeout = TIMEOUT_DMAMATCH;
    int f;
    unsigned int addr;
    int w;

    pc_outb (dp->iobase + REMOTEBYTECOUNT0, len & 0xff);
    pc_outb (dp->iobase + REMOTEBYTECOUNT1, len >> 8);
    pc_outb (dp->iobase + REMOTESTARTADDRESS0, 0x00);
    pc_outb (dp->iobase + REMOTESTARTADDRESS1, page);
    pc_outb (dp->iobase, NIC_REM_WRITE | NIC_START);

    if (dp->wordlength == 2) {
	for (w = 0; w < (len >> 1); w++) {
	    pc_outw (dp->iobase + NE_DATA, ((unsigned short *) buf)[w]);
	}
	if (len & 0x01) {
	    short tmp = buf[len - 1];	// so we can output a whole 16-bits

	    // if buf were on the end of something, we would die.
	    pc_outw (dp->iobase + NE_DATA, tmp);
	}
    }
    else {
	for (f = 0; f < len; f++) {
	    pc_outb (dp->iobase + NE_DATA, ((unsigned char *) buf)[f]);
	}
    }

    for (f = 0; f < timeout; f++) {
	unsigned int high = pc_inb (dp->iobase + REMOTESTARTADDRESS1);
	unsigned int low = pc_inb (dp->iobase + REMOTESTARTADDRESS0);

	addr = (high << 8) + low;
	if (
	    ((((page
		<< 8) + (dp->wordlength == 2 &&
			 (len & 0x01)) ? len : len + 1)) & 0xff) == low)
	    break;
    }
    if (f >= timeout) {
        // remote DMA address invalid expected page << 8, actual is addr
    }
    pc_outb (dp->iobase + INTERRUPTSTATUS, ISR_RDC);	// finish DMA cycle 
}

// -------------------------------------------------------------------------
