/* Copyright (c) 1987, 1988  Stanley T. Shebs. */
/* Copyright (c) 1995 Michael J. Peters */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact. */

/* This file contains almost all command functions. */
/* Help commands are in a separate file. */

#include "config.h"
#include "misc.h"
#include "dir.h"
#include "period.h"
#include "side.h"
#include "unit.h"
#include "map.h"
#include "mplay.h"
#include "X11.h"
#include "Motif.h"
#include "global.h"

bool tmparea = TRUE;    /* true for area painting, false for bar painting */

/* Make dialog go away. Normal dialog closure. */

void remove_dialog(Side *side) {

    if (dialog_active(side)) {
        destroy_dialog(side);
        add_move_callbacks(side);
    }
}

/* Make dialog go away. Also cancel any input callbacks from map_area */

void cancel_dialog(Widget any, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    if (dialog_active(side)) {
        if (Debug) printf("Canceling %s dialog\n", side->name);
        remove_map_callbacks(side);
        destroy_dialog(side);
        add_move_callbacks(side);
    }
}

/* Make side dialog go away. Normal side dialog closure. */

void remove_side_dialog(Side *side) {

    Sides_status *side_w;

    if (side_dialog_active(side)) {
        for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next)
            side_w->selected = FALSE;
        destroy_side_dialog(side);
        set_menu_sensitivity(side->side_menu, True);
    }
}

/* Make side dialog go away. Also reset side status' */

void cancel_side_dialog(Widget any, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Sides_status *side_w;

    if (side_dialog_active(side)) {
        for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next)
            side_w->selected = FALSE;
        destroy_side_dialog(side);
        set_menu_sensitivity(side->side_menu, True);
    }
}

Unit *current_unit(Side *side) {

    Unit *unit = side->curunit;

    if (unit == NULL)
        cmd_error(side, "No unit to operate on here!");

    return unit;
}       

/* Restore the saved "cur" slots. */

void restore_cur(Side *side) {

    side->curx = side->tmpcurx;  side->cury = side->tmpcury;
    side->curunit = side->tmpcurunit;
}

/* Survey mode allows player to look around (and change things) by moving */
/* cursor.  The same command toggles in and out, so need a case statement. */
/* Players waiting their turn will be in survey mode, but can't get out. */

void do_survey_mode(Widget sidemode, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    switch (side->mode) {
    case MOVE:
	survey_mode(side);
	break;
    case SURVEY:
	if (can_move_unit(side,side->curunit))
	  side->movunit = side->curunit;
	move_mode(side);
	break;
    default:
	case_panic("mode", side->mode);
    }
}

/* True if unit is adjacent to an unfriendly. */

bool adj_enemy(Unit *unit) {

    int d, x, y;
    viewdata view;

    for_all_directions(d) {
        x = wrap(unit->x + dirx[d]);  y = unit->y + diry[d];
        if (!neutral(unit)) {
            view = side_view(unit->side, x, y);
            if (view != EMPTY && enemy_side(side_n(vside(view)), unit->side))
                return TRUE;
        }
    }
    return FALSE;
}

/* Put unit to sleep for n turns.  If we sleep it next to something that */
/* might wake it up, then adjust flags so it won't wake up on next turn. */

void do_sentry(Side *side, Unit *unit, int n) {

  enter_procedure("do_sentry");
    if (side->teach) {
        cache_sentry(side, n);
    } else {
        order_sentry(unit, n);
        if (n > 1 && adj_enemy(unit))
            unit->orders.flags &= ~(ENEMYWAKE|NEUTRALWAKE);
    }
  exit_procedure();
}

void do_sentry_iterate(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    do_sentry(side, unit, side->itertime);
    show_status(side, unit);
}

/* Don't move for remainder of turn, but be awake next turn.  This also */
/* hooks into terrain painting, since the space bar is big and convenient. */

void do_sit(Widget any, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if (side->mode == SURVEY && Build) {
/*        paint_terrain(side); */
    } 
    else {
        if ((unit = current_unit(side)) == NULL) return;

        do_sentry(side, unit, 1);
        show_status(side, unit);
    }
}

void do_wake(Side *side, bool wakeocc) {

    Unit *unit;

    enter_procedure("do_wake");

    if (side->teach)
        cache_awake(side);
    else {
        if ((unit = current_unit(side)) == NULL) return;

        wake_unit(unit, wakeocc, WAKEFULL, (Unit *) NULL);
        show_status(side, unit);
    }

    exit_procedure();
}

/* Wake current unit */

void do_wake_it(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    do_wake(side, FALSE);
}

/* Wake current unit and occupants */

void do_wake_all(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    do_wake(side, TRUE);
}

void get_map_position(Widget map_area, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    XmDrawingAreaCallbackStruct *cbs =
        (XmDrawingAreaCallbackStruct *)call_data;

    if (cbs->reason == XmCR_INPUT) {
        if (cbs->event->xany.type == ButtonPress) {
            side->reqx = (int)cbs->event->xbutton.x;
            side->reqy = (int)cbs->event->xbutton.y;
        } else
            return;
    }
}           

/* The area wakeup. */

void wake_at(int x, int y) {

    Unit *unit = unit_at(x, y);

    enter_procedure("wake_at");
    if (unit != NULL || Build)
        wake_unit(unit, FALSE, WAKEFULL, (Unit *) NULL);
    exit_procedure();
}

void x_wake_area(Widget dialog, XtPointer client_data, XtPointer call_data) {

    static int xp, yp;
    int radius;
    Side *side = (Side *)client_data;

    XtRemoveCallback(side->map_area, XmNinputCallback, get_map_position, (XtPointer)side);

    deform(side, side->reqx, side->reqy, &xp, &yp);

    if (xp == side->curx && yp == side->cury) {
        if (side->curunit) do_wake(side, FALSE);

    } else {
        radius = distance(side->curx, side->cury, xp, yp);
        apply_to_area(side->curx, side->cury, radius, wake_at);
        if (side->curunit != NULL)
            show_status(side, side->curunit);
    }
    remove_dialog(side);
}

/* Wake units within an area */

void do_wake_area(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    remove_move_callbacks(side);
    
    side->reqx = side->curx;
    side->reqy = side->cury;

    create_map_dialog(side, "define radius of area to wake", "wake_area", 
        get_map_position, x_wake_area);
}

void do_map(Widget map_area, XtPointer client_data, XtPointer call_data) {

    int sx, sy;
    static int xp, yp;
    Side *side = (Side *)client_data;
    Unit *unit = side->curunit;
    XmDrawingAreaCallbackStruct *cbs =
        (XmDrawingAreaCallbackStruct *)call_data;

    if (cbs->reason == XmCR_INPUT) {
        if (cbs->event->xany.type == ButtonPress) {
            if (cbs->event->xbutton.button == Button1) {
                sx = (int)cbs->event->xbutton.x;
                sy = (int)cbs->event->xbutton.y;
            }
            else if (cbs->event->xbutton.button == Button2) {
                do_survey_mode(map_area, client_data, call_data);
                return;
            }
        } else
            return;
    }            

    deform(side, sx, sy, &xp, &yp);

    if (xp == side->curx && yp == side->cury) {
        if (unit) do_sit(NULL, (XtPointer)side, NULL);

    } else {
        if (side->teach) {
            cache_moveto(side, xp, yp);
        } else {
            switch (side->mode) {
            case MOVE:
                if (unit) {
                    order_moveto(unit, xp, yp);
                    unit->orders.flags &= ~SHORTESTPATH;
                }
                break;
            case SURVEY:
                move_survey(side, xp, yp);
                break;
            default:
                case_panic("mode", side->mode);
            }
        }
    }
}

/* Prompt for a type of a unit from player, maybe only allowing some types */
/* to be accepted.  Also allow specification of no unit type.  We do this */
/* by scanning the vector, building a string of chars and a vector of */
/* unit types, so as to be able to map back when done. */

void x_product(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;
    short u;

    if ((unit = current_unit(side)) == NULL) return;

    for_all_unit_types(u) {
        if (dialog_action == side->dialog_action[u])
            break;
    }     

    if (u != unit->product) {
        set_product(unit, u);
        set_schedule(unit);
    }

    show_product(side, unit);
    remove_dialog(side);
}

/* The command proper. */

void do_product(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    if (!global.setproduct)
        cmd_error(side, "No construction changes allowed in this game!");
    else {
        if (!can_produce(unit))
            cmd_error(side, "This unit can't build anything!");
        else {
            if (utypes[unit->type].occproduce ||
                utypes[unit->type].maker ||
                (unit->transport == NULL)) {

                if (!utypes[unit->type].maker)
                    wake_unit(unit, FALSE, WAKEOWNER, (Unit *) NULL);
                request_new_product(unit);
            } 
            else
                cmd_error(side, "This unit unable to produce inside other units.");
        }
    }
}

/* This is called when production is to be set or changed.  This
   routine allows the user to specify what will be produced after the
   current production is finished. */

void x_next_product(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;
    short u;

    if ((unit = current_unit(side)) == NULL) return;

    for_all_unit_types(u) {
        if (dialog_action == side->dialog_action[u])
            break;
    }     

    if (u != unit->product)
        unit->next_product = u;
    else 
        unit->next_product = unit->product;

    show_product(side, unit);
    remove_dialog(side);
}

void request_next_product(Unit *unit) {

    int u;
    Side *us = unit->side;

    if (humanside(us)) {
        sprintf(spbuf, "%s's next product will be: ", unit_handle(us, unit));
        u = ask_unit_type(us, utypes[unit->type].make);
        if (u < 0) {
            remove_move_callbacks(us);
            create_unit_dialog(us, spbuf, unit->product, x_next_product);
            make_current(us, unit);
        } else {
          if (u != unit->product  && u != NOTHING) {
            set_product(unit, u);
            set_schedule(unit);
          }
        }
    } else {
       /* the machine should never do this.  Just ignore it.  */
    }
}

/* The command proper. */

void do_next_product(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    if (!global.setproduct)
        cmd_error(side, "No construction changes allowed in this game!");
    else {
        if (!can_produce(unit))
            cmd_error(side, "This unit can't build anything!");
        else {
            if (utypes[unit->type].occproduce || 
                utypes[unit->type].maker ||
                (unit->transport == NULL)) {

                if (!utypes[unit->type].maker)
                    wake_unit(unit, FALSE, WAKEOWNER, (Unit *) NULL);
                request_next_product(unit);
            }
            else
                cmd_error(side, "This unit unable to produce inside other units.");
        }
    }
}

/* Set a unit to not produce anything (yes, this really is useful). */

void do_idle(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    if (!global.setproduct)
        cmd_error(side, "No production changes allowed in this scenario!");
    else {
        set_product(unit, NOTHING);
        unit->schedule = side->itertime;
        show_product(side, unit);
    }
}


/* Search for a friendly refueler within range and set course for it.  */

void do_return(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if (side->teach)
        cache_return(side, side->itertime);
    else {
        if ((unit = current_unit(side)) == NULL) return;

        order_return(unit, side->itertime);
        unit->orders.flags &= ~(SUPPLYWAKE|ENEMYWAKE|NEUTRALWAKE);
    }
}

void show_map_position(Widget map_area, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    XmDrawingAreaCallbackStruct *cbs =
        (XmDrawingAreaCallbackStruct *)call_data;

    if (cbs->reason == XmCR_INPUT) {
        if (cbs->event->xany.type == ButtonPress) {
            side->reqx = (int)cbs->event->xbutton.x;
            side->reqy = (int)cbs->event->xbutton.y;
            deform(side, side->reqx, side->reqy, &(side->curx), &(side->cury));
            make_current(side, unit_at(side->curx, side->cury));
        } else
            return;
    }
}           

/* Set unit to move to a given location.  */

void x_moveto(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    XtRemoveCallback(side->map_area, XmNinputCallback, show_map_position, (XtPointer)side);

    if (side->tmpcurx == side->curx && side->tmpcury == side->cury) {
        if (side->tmpcurunit) do_sit(NULL, (XtPointer)side, NULL);

    } else {
        if (side->teach) {
            cache_moveto(side, side->curx, side->cury);
        } else if (Build) {
            leave_hex(side->tmpcurunit);
            occupy_hex(side->tmpcurunit, side->curx, side->cury);
            make_current(side, side->tmpcurunit);
            all_see_hex(side->curx, side->cury);
        } else {
          order_moveto(side->tmpcurunit, side->curx, side->cury);
        }
    }
    restore_cur(side);        
    show_status(side, side->curunit);
    remove_dialog(side);
}

/* The command proper. */

void do_moveto(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    remove_move_callbacks(side);
    side->tmpcurx = side->curx;  side->tmpcury = side->cury;
    side->tmpcurunit = unit;
    create_map_dialog(side, "define final destination", "move_to_position", 
        show_map_position, x_wake_area);
}

/* Set orders to follow a leader unit. */

void x_follow(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Unit *leader;
    Side *side = (Side *)client_data;

    XtRemoveCallback(side->map_area, XmNinputCallback, show_map_position, (XtPointer)side);

    if ((leader = unit_at(side->curx, side->cury)) != NULL &&
        leader->side == side) {

        if (leader != side->tmpcurunit)
            if (side->teach)
                cache_follow(side, leader, side->itertime);
            else
                order_follow(side->tmpcurunit, leader, side->itertime);
        else
            cmd_error(side, "Unit can't follow itself!");
    }
    else
        cmd_error(side, "No unit to follow!");

    restore_cur(side);        
    show_status(side, side->curunit);
    remove_dialog(side);
}

/* The command proper. */

void do_follow(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    remove_move_callbacks(side);
    side->tmpcurx = side->curx;  side->tmpcury = side->cury;
    side->tmpcurunit = unit;
    create_map_dialog(side, "select unit to follow", "follow_leader", 
        show_map_position, x_follow);
}

/* Set unit to attempt to follow a coast.  Needs a starting direction, */
/* which can be computed from a position. */

void x_coast(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    int dir;
    int ccw = 0;
    int count = 0;

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    for (dir = 0; dir < NUMDIRS; dir++) {
        if (dialog_action == side->dialog_action[dir])
            break;
    }

    if (!could_move_in_dir(unit, dir)) {
        cmd_error(side, "%s can't move there!", unit_handle(side, unit));
    } else {
        if (could_move_in_dir(unit, left_of(dir)))
            ccw++; count++;
        if (could_move_in_dir(unit, right_of(dir)))
            ccw--; count++;
        if (count>0 && ccw==0)
            cmd_error(side, "Those hexes don't share an edge!");
        else {
            notify(side, "%s will follow %s%s",
                   unit_handle(side, unit),
                   (ccw==0)?"isthsmus":"edge ",
                   (ccw==0)?"":(ccw>0)?"right":"left");

            if (side->teach)
                cache_edge(side, dir, ccw, side->itertime);
            else
                order_edge(unit, dir, ccw, side->itertime);
        }
    }

    remove_dialog(side);
}

/* The command proper. */

void do_coast(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    if (current_unit(side) == NULL) return;

    remove_move_callbacks(side);
    create_dir_dialog(side, "Select direction to follow coast");
}

/* Patrolling goes back and forth between two points.  First point is the */
/* current position. */

void x_patrol(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    XtRemoveCallback(side->map_area, XmNinputCallback, show_map_position, (XtPointer)side);

    if (side->tmpcurx == side->curx && side->tmpcury == side->cury) {
        if (side->tmpcurunit) do_sit(NULL, (XtPointer)side, NULL);

    } else {
        if (side->teach) {
            cache_patrol(side, side->sounit->x, side->sounit->y,
                         side->curx, side->cury, side->itertime);
        } else {
            order_patrol(side->tmpcurunit, side->tmpcurunit->x, side->tmpcurunit->y,
                         side->curx, side->cury, side->itertime);
        }
    }

    restore_cur(side);
    show_status(side, side->curunit);
    remove_dialog(side);
}

/* The command proper. */

void do_patrol(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    remove_move_callbacks(side);
    side->tmpcurx = side->curx;  side->tmpcury = side->cury;
    side->tmpcurunit = unit;
    create_map_dialog(side, "define other patrol endpoint", "patrol", 
        show_map_position, x_patrol);
}

/* Delay a unit's move until a later time.  The set flag will be recognized */
/* by the movement loops, when deciding which unit to move next. */

void do_delay(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    if (side->mode == MOVE) {
        delete_unit(unit);
        insert_unit_tail(side->unithead, unit);
        unit->orders.type = DELAY;
        side->movunit = NULL;
        notify(side, "Delaying moving %s.", unit_handle(side, unit));
    }
}

void x_unit_name(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;
    XmSelectionBoxCallbackStruct *cbs =
        (XmSelectionBoxCallbackStruct *)call_data;
    char *name;

    if ((unit = current_unit(side)) == NULL) return;

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &name) ||
       strlen(name) == 0) 
        notify(side, "Name not changed.");
    else {
        unit->name = copy_string(name);
        show_info(side);
    }
    remove_dialog(side);
}

void do_unit_name(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    if (current_unit(side) == NULL) return;

    remove_move_callbacks(side);
    create_prompt_dialog(side, "New name for unit:", "new_unit_name",
        x_unit_name);
}

/* Reclaim both the unit's supplies and anything used in its making, but */
/* only let a maker of the unit reclaim its ingredients. */

void recycle_unit(Unit *unit, Unit *unit2) {

    int u = unit->type, u2 = unit2->type, r, scrap;

    for_all_resource_types(r) {
        transfer_supply(unit, unit2, r, unit->supply[r]);
        if (could_make(u2, u) > 0) {
            scrap = utypes[u].tomake[r];
            unit2->supply[r] += (scrap * period.efficiency) / 100;
        }
    }
}

/* Get rid of unit.  Some units cannot be disbanded, but if they can, the */
/* resources go to a transport if one is there.  Disbanding a transport */
/* also disbands all the occupants - oh well. */

void do_disband(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    enter_procedure("do_disband");
    if ((unit = current_unit(side)) == NULL) return;

    if (utypes[unit->type].disband || Build) {
        notify(side, "%s goes home.", unit_handle(side, unit));
        if (unit->transport != NULL) recycle_unit(unit, unit->transport);
        kill_unit(unit, DISBAND);
        make_current(side, unit_at(side->curx, side->cury));
    } else
        cmd_error(side, "You can't just get rid of the %s!", utypes[unit->type].name);

    exit_procedure();
}

/* Center the screen on the current location. */

void do_recenter(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    recenter(side, side->curx,side->cury);
    if (Debug) notify(side, "Current location %d %d ",  side->curx,side->cury);
}

/* This is a clever (if I do say so myself) command to examine all occupants */
/* and suboccupants, in a preorder fashion. */

void do_occupant(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *nextup;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    switch (side->mode) {
    case MOVE:
        cmd_error(side, "Can only look at occupants when in survey mode!");
        break;
    case SURVEY:
        if (unit->occupant != NULL) {
            make_current(side, unit->occupant);
        } else if (unit->nexthere != NULL) {
            make_current(side, unit->nexthere);
        } else {
            nextup = unit->transport;
            if (nextup != NULL) {
                while (nextup->transport != NULL && nextup->nexthere == NULL) {
                    nextup = nextup->transport;
                }
                if (nextup->nexthere != NULL)
                    make_current(side, nextup->nexthere);
                if (nextup->transport == NULL)
                    make_current(side, nextup);
            }
        }
        break;
    default:
        case_panic("mode", side->mode);
        break;
    }
}

/* Marking is for the purpose of rearranging units within a hex. */

void do_mark_unit(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    if (utypes[unit->type].holdvolume == 0 ) {
        cmd_error(side, "%s cannot be transports", utypes[unit->type].name);
        return;
    }        
    side->markunit = unit;
    notify(side, "%s has been marked.", unit_handle(side, unit));
}

void do_embark_unit(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    do_embark(side, unit, 0);
}

/* Give supplies to a transport.  The argument tells how many to give. */

void x_give(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;
    Unit *main;
    char *quantity;
    int qty;
    bool something = FALSE;
    int u, m, r, gift, actual;
    XmSelectionBoxCallbackStruct *cbs =
        (XmSelectionBoxCallbackStruct *)call_data;

    if ((unit = current_unit(side)) == NULL) return;
    u = unit->type;

    main = unit->transport;
    if (main == NULL) {
        cmd_error(side, "Can't transfer supplies here!");
        return;
    }

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &quantity) ||
       strlen(quantity) == 0 ||
       sscanf(quantity, "%d", &qty) == 0) 
        qty = 0;

    sprintf(spbuf, "");
    m = main->type;
    for_all_resource_types(r) {
        gift = (qty < 0 ? utypes[m].storage[r] - main->supply[r] : qty);
        if (gift > 0) {
            something = TRUE;
            /* Be stingy if we're low */
            if (2 * unit->supply[r] < utypes[u].storage[r])
                gift = max(1, gift/2);
            actual = transfer_supply(unit, main, r, gift);
            sprintf(tmpbuf, " %d %s", actual, rtypes[r].name);
            strcat(spbuf, tmpbuf);
        }
    }

    if (something)
        notify(side, "%s gave%s.", unit_handle(side, unit), spbuf);
    else
        notify(side, "%s gave nothing.", unit_handle(side, unit));

    show_resources(side, unit);
    remove_dialog(side);
}

void do_give(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    if (current_unit(side) == NULL) return;

    remove_move_callbacks(side);
    create_prompt_dialog(side, "Enter quantity of supplies to give", "give_supplies",
        x_give);
}

void x_take(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;
    char *quantity;
    int qty;
    XmSelectionBoxCallbackStruct *cbs =
        (XmSelectionBoxCallbackStruct *)call_data;

    if ((unit = current_unit(side)) == NULL) return;

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &quantity) ||
       strlen(quantity) == 0 ||
       sscanf(quantity, "%d", &qty) == 0) 
        qty = -1;

    take_it(side, unit, qty);

    remove_dialog(side);
}

void do_take(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    if (current_unit(side) == NULL) return;

    remove_move_callbacks(side);
    create_prompt_dialog(side, "Enter quantity of supplies to take", "take_supplies",
        x_take);
}

/* Give a unit to another side (possibly to neutrals).  Units that won't */
/* change their sides when captured won't change voluntarily either. */

void x_give_unit(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Side *side2;
    Unit *unit;
    int u, n=0;

    if ((unit = current_unit(side)) == NULL) return;
    u = unit->type;

    for_all_sides(side2) {
        if (side != side2)
            if (dialog_action == side->dialog_action[n]) {
                break;
            }
            n++;
    }

    if (utypes[u].changeside || Build) {
        unit_changes_side(unit, side2, CAPTURE, GIFT);
        all_see_hex(unit->x, unit->y);
        if (global.setproduct) {
          set_product(unit, NOTHING);
          unit->schedule = 0;
        }
    } 
    else
        cmd_error(side, "You can't just give away the %s!", utypes[u].name);

    remove_dialog(side);
}

void do_give_unit(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    if (current_unit(side) == NULL) return;

    remove_move_callbacks(side);
    create_give_dialog(side, "Select side to give unit to:", x_give_unit);
}

/* Designate the current location as the center of action and sort all */
/* of our own units relative to it. */

void do_center(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    if (current_unit(side) == NULL) return;
    side->cx = side->curx;  side->cy = side->cury;
    sort_side(side, TRUE);
    set_side_movunit(side);
    notify(side, "Units reordered.");
}

/* Send a short (1 line) message to player(s). */

void x_message(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Sides_status *side_w;    
    XmSelectionBoxCallbackStruct *cbs =
        (XmSelectionBoxCallbackStruct *)call_data;
    char *msg;
    bool somebody=FALSE;

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &msg) ||
       strlen(msg) == 0)
        notify(side, "You keep your mouth shut.");
    else {
        for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next) {
            if (side_w->selected) {
                somebody = TRUE;
                notify(side_w->side, "The %s say to you: %s",
                       plural_form(side->name), msg);
            }
        }

        if (!somebody)
            notify_all("The %s announce: %s", plural_form(side->name), msg);
    }

    if (somebody)
        for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next)
            side_w->selected = FALSE;

    remove_side_dialog(side);
}

void x_sidelist(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Side *side2;
    Sides_status *side_w;
    int n = 0;
    XmToggleButtonCallbackStruct *cbs =
        (XmToggleButtonCallbackStruct *)call_data;

    for_all_sides(side2) {
        if (side != side2) {
            if (dialog_action == side->side_dialog_action[n])
                break;
            n++;
        }
    }

    spbuf[0] = '\0';
    for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next) {
        if (side_w->side == side2)
            side_w->selected = cbs->set;

        if (side_w->selected) {
            if (spbuf[0] == '\0')
                sprintf(spbuf, "Say to the %s", plural_form(side_w->side->name));
            else {
                sprintf(tmpbuf, ", %s", plural_form(side_w->side->name));
                strcat(spbuf, tmpbuf);
            }
        }
    }
    if (spbuf[0] == '\0')
        sprintf(spbuf, "Broadcast:", plural_form(side_w->side->name));
    else
        strcat(spbuf, ":");

    XtVaSetValues(side->side_dialog,
        XtVaTypedArg, XmNselectionLabelString, XtRString,
            spbuf, strlen(spbuf),
        NULL);
}

void x_sidelist2(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Side *side2;
    Sides_status *side_w;
    int n = 0;
    XmToggleButtonCallbackStruct *cbs =
        (XmToggleButtonCallbackStruct *)call_data;

    for_all_sides(side2) {
        if (side != side2) {
            if (dialog_action == side->side_dialog_action[n])
                break;
            n++;
        }
    }

    for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next) {
        if (side_w->side == side2) {
            side_w->selected = cbs->set;
            break;
        }
    }
}

void do_message(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_msg_dialog(side);
}

void x_war(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Side *side2 = (Side *)client_data;
    Sides_status *side_w;    
    bool somebody=FALSE;

    for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next) {
        if (side_w->selected) {
            somebody = TRUE;
            notify(side, "You declare war on the %s!",
                   plural_form(side_w->side->name));
            declare_war(side, side_w->side);
            for_all_sides(side2) {
	        draw_all_sides(side);
                show_map(side2);
	        show_world(side);
	    }
        }
    }

    if (!somebody)
        notify(side, "You keep your mouth shut.");
    else
        for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next)
            side_w->selected = FALSE;

    remove_side_dialog(side);
}

void do_war(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_declare_dialog(side, "declare_war", "Select side(s) to declare war with:", 
        x_war);
}

void x_neutral(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Side *side2 = (Side *)client_data;
    Sides_status *side_w;    
    bool somebody=FALSE;

    for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next) {
        if (side_w->selected) {
            somebody = TRUE;
            notify(side, "You propose neutrality with the %s.",
                   plural_form(side_w->side->name));
            side->attitude[side_number(side_w->side)] = NEUTRAL;
            if (side_w->side->attitude[side_number(side)] == NEUTRAL) {
                declare_neutrality(side, side_w->side);
                for_all_sides(side2) {
	            draw_all_sides(side);
                    show_map(side2);
	            show_world(side);
	        }
            } else {
                notify(side_w->side, "The %s propose neutrality.",
                       plural_form(side->name));
	    }
        }
    }

    if (!somebody)
        notify(side, "You keep your mouth shut.");
    else
        for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next)
            side_w->selected = FALSE;

    remove_side_dialog(side);
}

void do_neutral(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_declare_dialog(side, "declare_neutrality", "Select side(s) to declare neutrality with:", 
        x_neutral);
}

void x_alliance(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Side *side2 = (Side *)client_data;
    Sides_status *side_w;    
    bool somebody=FALSE;

    for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next) {
        if (side_w->selected) {
            somebody = TRUE;
            notify(side, "You propose a formal alliance with the %s.",
                   plural_form(side_w->side->name));
            side->attitude[side_number(side_w->side)] = ALLY;
            if (side_w->side->attitude[side_number(side)] >= ALLY) {
                declare_alliance(side, side_w->side);
                for_all_sides(side2) {
	            draw_all_sides(side);
                    show_map(side2);
	            show_world(side);
	        }
            } else {
                notify(side_w->side, "The %s propose a formal alliance.",
                       plural_form(side->name));
	    }
        }
    }

    if (!somebody)
        notify(side, "You keep your mouth shut.");
    else
        for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next)
            side_w->selected = FALSE;

    remove_side_dialog(side);
}

void do_alliance(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_declare_dialog(side, "declare_alliance", "Select side(s) to declare alliance with:", 
        x_alliance);
}

void x_briefing(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Sides_status *side_w;    
    bool somebody=FALSE;

    for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next) {
        if (side_w->selected) {
            somebody = TRUE;
            notify(side_w->side, "Receiving a briefing from the %s...",
                   plural_form(side->name));
            reveal_side(side, side_w->side, 100);
            notify(side, "You just briefed the %s on your position.",
                   plural_form(side_w->side->name));
        }
    }

    if (!somebody)
        notify(side, "You keep your mouth shut.");
    else
        for (side_w = side->sideslist; side_w != NULL; side_w = side_w->next)
            side_w->selected = FALSE;

    remove_side_dialog(side);
}

void do_briefing(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_declare_dialog(side, "send_briefing", "Select side(s) to send briefing:", 
        x_briefing);
}

/* Take the current player out of the game while letting everybody else */
/* continue on. */

void x_resign(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    resign_game(side, NULL);
}

void do_resign(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_game_dialog(side, "Do you really want to give up?", "resign", 
        x_resign);
}

void x_lost(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    close_display(side);
}


void x_exit(Widget menu, XtPointer client_data, XtPointer call_data) {

    exit_xconq();
}

/* Stuff game state into a file.  By default, it goes into the current */
/* directory.  If building a scenario, will ask about each section, values */
/* of globals, and dest file before actually writing anything. */
/* No capability to write out period at present... */

void x_save_build(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    char *sects, *fname;
    int sdetail = 1, udetail = 1;

    fname = "random.scn";
    sprintf(spbuf, "------");
    if (iindex('v', sects) >= 0) spbuf[0] = '+';
    if (iindex('p', sects) >= 0) spbuf[1] = '+';
    if (iindex('m', sects) >= 0) spbuf[2] = '+';
    if (iindex('g', sects) >= 0) spbuf[3] = '+';
    if (iindex('s', sects) >= 0) spbuf[4] = '+';
    if (iindex('u', sects) >= 0) spbuf[5] = '+';
    if (iindex('s', sects) >= 0) {
        sdetail = 1;
        if (isdigit(sects[iindex('s', sects)+1]))
            sdetail = sects[iindex('s', sects)+1] - '0';
    }
    if (iindex('u', sects) >= 0) {
        udetail = 1;
        if (isdigit(sects[iindex('u', sects)+1]))
            udetail = sects[iindex('u', sects)+1] - '0';
    }
    notify(side, "Mapfile with sections %s will be saved to \"%s\" ...",
           spbuf, fname);
    if (write_scenario(fname, spbuf, sdetail, udetail)) {
        notify(side, "Done writing to \"%s\".", fname);
    } else
        cmd_error(side, "Can't open file \"%s\"!", fname);
}

/* Make a header appropriate to a save file, write the file, and leave. */

void x_save_game(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    remove_side_dialog(side);
    printf("Game will be saved to \"%s\" ...\n", SAVEFILE);
    notify_all("Game will be saved to \"%s\" ...", SAVEFILE);
    if (write_savefile(SAVEFILE)) {
        close_displays();
        exit(0);
    } else
        cmd_error(side, "Can't open file \"%s\"!", SAVEFILE);
}

/* The command proper just sets up different handlers, depending on */
/* whether we're building (and therefore saving a scenario/fragment), or */
/* saving as much game state as possible, for resumption later. */

void do_save(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    if (Build) {
        create_game_dialog(side, "Sections to write?", "save_build", 
            x_save_build);
    }
    else {
        create_game_dialog(side, "You really want to save?", "save_game", 
            x_save_game);
    }
}

void x_rename(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Side *side2;
    XmSelectionBoxCallbackStruct *cbs =
        (XmSelectionBoxCallbackStruct *)call_data;
    char *name;

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &name) ||
       strlen(name) == 0) 
        notify(side, "Name not changed.");
    else {
        side->name = copy_string(name);
        for_all_sides(side2) {
            if (active_display(side2))
                draw_all_sides(side2);
        }
    }
    remove_side_dialog(side);
}

void do_rename(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_options_dialog(side, "New name for yourself:", "name", x_rename);
}

void x_beeptime(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    XmSelectionBoxCallbackStruct *cbs =
        (XmSelectionBoxCallbackStruct *)call_data;
    char *beepbuf;
    int beeptime;

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &beepbuf) ||
       strlen(beepbuf) == 0 ||
       sscanf(beepbuf, "%d", &beeptime) == 0)
        notify(side, "Beep time not changed.");
    else
        side->startbeeptime = beeptime;

    remove_side_dialog(side);
}

void do_beeptime(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_options_dialog(side, "Beep time (seconds):", "beep_time", x_beeptime);
}

void x_itertime(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    XmSelectionBoxCallbackStruct *cbs =
        (XmSelectionBoxCallbackStruct *)call_data;
    char *turns;
    int itertime;

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &turns) ||
       strlen(turns) == 0 ||
       sscanf(turns, "%d", &itertime) == 0)
        notify(side, "Order iteration not changed.");
    else
        side->itertime = itertime;

    remove_side_dialog(side);
}

void do_itertime(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_options_dialog(side, "Order iteration (turns):", "order_iteration", x_itertime);
}

void x_background(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    XmSelectionBoxCallbackStruct *cbs =
        (XmSelectionBoxCallbackStruct *)call_data;
    char *colorname;
    Pixel bgcolor;

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &colorname) ||
        strlen(colorname) == 0 || 
        ((bgcolor = request_color(side, colorname)) == white_color(side)))
        notify(side, "Background not changed.");
    else
        change_bgcolor(side, bgcolor);

    remove_side_dialog(side);
}

void do_background(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    create_options_dialog(side, "Background color:", "background", x_background);
}

void x_showmode(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    int showmode;

    for(showmode = 0; showmode < NUMSHOWMODES; showmode++)
        if (dialog_action == side->side_dialog_action[showmode])
            break;

    side->appdata.showmode = showmode;
    show_map(side);

    remove_side_dialog(side);
}

void do_showmode(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    set_menu_sensitivity(side->side_menu, False);
    create_showmode_dialog(side);
}

/* Morph just like a Power Ranger */

void x_morph(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    side->humanp = !side->humanp;
    if (side->humanp) {
        numhumans++;
        add_move_callbacks(side);
    }   
    else {
        numhumans--;
        remove_move_callbacks(side);       
    }   
    for_all_side_units(side, unit)
        unit->area = area_index(unit->x, unit->y);
/*    init_sighandlers(); */
    remove_side_dialog(side);
}

void do_morph(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    set_menu_sensitivity(side->side_menu, False);
    sprintf(spbuf, "Do you really want to become a %s?", (side->humanp? "machine" : "human"));
    create_game_dialog(side, spbuf, "metamorphose", 
        x_morph);
}

/* This command provides a short note about the current hex.  It is */
/* useful as a supplement to the general help command. */

void do_ident(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    viewdata view = side_view(side, side->curx, side->cury);
    char t = terrain_at(side->curx, side->cury);
    Side *side2;

    if (view == UNSEEN) {
        notify(side, "You see unexplored territory");
    } else if (view == EMPTY) {
        notify(side, "You see unoccupied %s", ttypes[t].name);
    } else {
        side2 = side_n(vside(view));
        notify(side, "You see a %s %s (in the %s)",
               (side2 == NULL ? "neutral" : side2->name),
               utypes[vtype(view)].name, ttypes[t].name);
    }
}

/* Flicker on the current position, in case it's not easily visible. */

void do_flash(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    int sx, sy;

    xform(side, unwrap(side, side->curx, side->cury), side->cury, &sx, &sy);
    flash_position(side, sx, sy, 1000);
}

/* Refresh map, erasing units we haven't seen in n turns */

void x_refresh(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    XmSelectionBoxCallbackStruct *cbs =
        (XmSelectionBoxCallbackStruct *)call_data;
    char *turns;
    int n;
    int x, y, ts;

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &turns) ||
       strlen(turns) == 0 ||
       sscanf(turns, "%d", &n) == 0) 
        n = 0;

#ifdef PREVVIEW
    if (n > 0) {  /* remove the views of units likely to have moved. */
      for_all_hexes(x, y)
          if (side_view_age(side, x, y) > n) {
            int viewunit = vtype(side_view(side, x, y));

            if (viewunit != EMPTY && viewunit != UNSEEN
                && mobile(viewunit)) {
              ts = side_view_timestamp(side, x, y);
              set_side_view(side, x, y, EMPTY);
              side_view_timestamp(side, x, y) = ts;
            }
          }
    }
#endif
    show_map(side);
    show_world(side);
    remove_dialog(side);
}

/* Refresh map, erasing units we haven't seen in n turns */

void do_refresh(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    remove_move_callbacks(side);
    create_prompt_dialog(side, "Enter turns:", "refresh",
        x_refresh);
}

/* Redraw map */

void do_redraw(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    show_map(side);
    show_world(side);
}

/* Find the next one of the given unit type. */

void x_flash_unit(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Specific_unit *firstunit = (Specific_unit *)client_data;
    Specific_unit *curunit = firstunit;
    XmSelectionBoxCallbackStruct *cbs = 
        (XmSelectionBoxCallbackStruct *)call_data;
    char *name;
    Unit *unit;
    Side *side = firstunit->unit->side;

    if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &name) ||
       strlen(name) == 0)
        return;
    while (curunit != NULL) {
        if (strcmp(short_unit_handle(curunit->unit), name) == 0) {
            unit = curunit->unit;
            remove_dialog(side);
            make_current(side, unit);
            break;
        }
        curunit = curunit->next;
    }
            
    remove_dialog(side);
}

void x_find_unit(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;
    short u;
    Specific_unit *firstunit = NULL;
    Specific_unit *newunit, *curunit;
    int itemno = 0;

    for_all_unit_types(u) {
        if (dialog_action == side->dialog_action[u])
            break;
    }     

    for_all_side_units(side, unit) {
        if (unit->type == u) {
            newunit = (Specific_unit *)malloc(sizeof(Specific_unit));
            newunit->unit = unit;
            newunit->itemno = ++itemno;
            newunit->next = NULL;
            if (firstunit == NULL)
                firstunit = newunit;
            else {
                curunit = firstunit;
                while (curunit->next != NULL) curunit = curunit->next;
                curunit->next = newunit;
            }
        }
    }
    if (firstunit != NULL)
        create_find_dialog(side, firstunit, itemno);

}

/* This command is useful for finding units in bases and a specific unit */
/* when a message mentioning is on the screen. */

void do_find_unit(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    int u;

    if (side->mode != SURVEY) {
        cmd_error(side, "Find unit only works in survey mode.");
        return;
    }

    remove_move_callbacks(side);

    for_all_unit_types(u)
        side->unitslist[u].bvec = TRUE;
    create_unit_dialog(side, "Find which type of unit?", NOTHING, x_find_unit);
}

/* Find next unit of the given type. */

void do_find_next_unit_by_type(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;
    int type;

    if (side->mode != SURVEY) {
        cmd_error(side, "Find unit only works in survey mode.");
        return;
    }

    if (side->curunit == NULL) {
        do_find_unit(menu, (XtPointer)side, call_data);
        return;
    }

    unit = side->curunit;
    type = unit->type; 

    for (;;) {
        if (unit != NULL) unit = unit->next;
        if (unit == NULL)
            unit = side->unithead;
        if (unit->side == side && alive(unit) && unit->type == type) {
            make_current(side, unit);
            return;
        }
    }
}

/* Determine how far away another point is.  */

void x_distance(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    static int xp, yp;
    int dist;

    enter_procedure("x_distance");

    XtRemoveCallback(side->map_area, XmNinputCallback, get_map_position, (XtPointer)side);

    deform(side, side->reqx, side->reqy, &xp, &yp);

    dist = distance(side->curx, side->cury, xp, yp);
    notify(side, "Distance from (%d, %d) to (%d, %d) is %d",
           side->curx, side->cury, xp, yp, dist);

    remove_dialog(side);
    exit_procedure();
}

/* The command proper. */

void do_distance(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    remove_move_callbacks(side);
    
    side->reqx = side->curx;
    side->reqy = side->cury;

    create_map_dialog(side, "distance to where?", "determine_distance", 
        get_map_position, x_distance);
}

/* order a unit to move toward the marked unit */

void x_movetounit(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    if (side->teach)
        cache_movetounit(side, side->markunit, side->itertime);
    else
        order_movetounit(side->tmpcurunit, side->markunit, side->itertime);

    restore_cur(side);
    show_status(side, side->curunit);
    remove_dialog(side);
}

void do_movetounit(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    if (side->markunit==NULL) {
        cmd_error(side, "No marked unit.  Use 'x' to mark the destination");
        return;
    }        
    if (!can_carry(side->markunit, unit)) {
        cmd_error(side, "%s can't carry us", unit_handle(side, side->markunit));
        return;
    }        

    remove_move_callbacks(side);
    sprintf(spbuf, "Move toward %s?", unit_handle(side, side->markunit));
    create_question_dialog(side, spbuf, "move_to_unit", x_movetounit);

    side->tmpcurx = side->curx;  side->tmpcury = side->cury;
    side->tmpcurunit = unit;
    move_survey(side, side->markunit->x, side->markunit->y);
}

void do_fill(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    enter_procedure("do_fill");
    if (side->teach)
        cache_fill(side, side->itertime);
    else {
        order_fill(unit, side->itertime);
        wake_if_full(unit);
        show_status(side, unit);
    }
    exit_procedure();
}

/* find the nearest filling transport and order unit towards it */

void do_movetotransport(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    enter_procedure("do_movetotransport");

    if (side->teach)
        cache_movetotransport(side, side->itertime);
    else {
        order_movetotransport(unit, side->itertime);
        show_status(side, unit);
    }
    exit_procedure();
}

/* Sentry until unit can embark */

void do_sentry_embark(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    if ((unit = current_unit(side)) == NULL) return;

    enter_procedure("do_sentry_embark");
    order_embark(unit, side->itertime);
    show_status(side, unit);
    exit_procedure();
}

/* A standing order is acquired by snarfing the next order and saving it */
/* rather than applying it to some unit. */

void get_standing_order(Side *side, int type) {

    int i;

    if (side->curunit->standing == NULL) {
        side->curunit->standing =
            (StandingOrder *) malloc(sizeof(StandingOrder));
        for_all_unit_types(i)
          side->curunit->standing->orders[i] = NULL;
    }
    side->teach = TRUE;
    side->soutype = type;
    side->tmporder = (Order *) malloc(sizeof(Order));
    side->tmporder->morder = FALSE;
    side->tmporder->flags = NORMAL;

    notify(side, "Next input order will become the standing order (DEL to delete).");
    show_mode(side);
}

/* Set standing orders for a unit of a given type that enters a given city. */
/* Space for standing orders is dynamically allocated the first time we */
/* request some orders. */

void x_standing_1(Widget dialog_action, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    int u;

    if (current_unit(side) == NULL) return;

    for_all_unit_types(u) {
        if (dialog_action == side->dialog_action[u])
            break;
    }     
    get_standing_order(side, u);

    remove_dialog(side);
}

/* The command proper starts the ball rolling by prompting for the type */
/* of unit that will get standing orders.  Of course, if a unit is not of */
/* a type that has occupants, standing orders are pretty useless.  Also, */
/* if only one type of occupant is possible, then no need to ask. */

void do_standing(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;
    int u;

    if ((unit = current_unit(side)) == NULL) return;

    if (Debug) printf("doing standing order\n");
    u = ask_unit_type(side, utypes[unit->type].capacity);
    if (Debug) printf("u is %d\n", u);
    if (u < 0) {
        show_standing_orders(side, unit);
        remove_move_callbacks(side);
        create_unit_dialog(side, "Type of occupant to get standing orders?", NOTHING, x_standing_1);
        side->sounit = unit;
    } 
    else if (u == NOTHING) {
        cmd_error(side, "This unit never has occupants to give orders to!");
    } 
    else {
        show_standing_orders(side, unit);
        get_standing_order(side, u);
    }
}

/* this command does nothing normally, but in teach mode it deletes the
   standing order */

void do_nothing(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

  if (side->teach) {
    free(side->tmporder);
    side->tmporder=NULL;
    finish_teach(side);
  }
}

#ifndef NO_DISEMBARK
/* Disembark is intended to complement embarking... */
/* This should, theoretically, be easier to implement than do_embark */

void do_disembark(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;
    Unit *mainunit;

    if ((unit = current_unit(side)) == NULL) return;
    mainunit = unit_at(unit->x, unit->y);

    enter_procedure("do_disembark");
    if (side->teach) /* setting a standing order? */
        cache_disembark(side, side->itertime);
    else {

        /* find a unit to occupy. In this case, the 'mainunit' */
        if (mainunit != unit) {
            if(can_carry(mainunit, unit)) {
                leave_hex(unit);
                occupy_hex(unit, mainunit->x, mainunit->y);
                show_info(side);
                routine(" done disembark");

                exit_procedure();
            } 
            else {
                cmd_error(side, "Can't disembark onto the unit here!!");

                exit_procedure();
            }
        }  
        else {
            cmd_error(side, "That unit can't exit itself!");
            exit_procedure();
        }
  }
  exit_procedure();
}
#endif

/* toggle followaction flag */

void do_look(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    side->followaction = !side->followaction;
    show_mode(side);
}

/* Set the unit to automatic control.  */

void do_auto(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    Unit *unit;

    enter_procedure("do_auto");
    if (side->teach)
        cache_auto(side);
    else {
        if ((unit = current_unit(side)) == NULL) return;
        unit->orders.morder = TRUE;
    }
    exit_procedure();
}

/* Find next dead unit */

void do_next_dead(Widget dialog, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;

    if (side->mode != SURVEY) {
        cmd_error(side, "Must be in survey mode to find dead units.");
        return;
    }

    flush_dead_units();
    if (side->deadunits == NULL) {
        notify(side, "No units dies in the last turn.");
        return;
    }

    if (side->curdeadunit == NULL || side->curdeadunit->next == NULL)
        side->curdeadunit = side->deadunits;
    else 
        side->curdeadunit = side->curdeadunit->next;

    if (side->curdeadunit != NULL)
        move_survey(side, side->curdeadunit->prevx,
                    side->curdeadunit->prevy);
}
