/***************************************************************************
 *   Copyright (C) 2005-2010 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QCheckBox>
#include <QLabel>
#include <QRadioButton>
#include <QDate>

#include <QComboBox>
#include <KConfig>
#include <KDialog>
#include <KGlobal>
#include <KIconLoader>
#include <KLineEdit>
#include <KLocale>
#include <KNumInput>
#include <KTabWidget>
#include <KUrlRequester>

#include "kdatecombo.h"
#include "komparatorfilefilter.h"
#include "komparatorurllist.h"
#include "komparatorwidget.h"

#include "komparatorpreset.h"

#include <version.h>

KomparatorPreset::KomparatorPreset( KomparatorWidget *_parent, KSharedConfigPtr &config )
{
	m_parent = _parent;
	m_preset_title = i18n( "default" );
	m_config = config;

	m_url1 = "";
	m_url2 = "";
	m_regex_contains = "*";
	m_regex_containsnot = "";
	m_size_selection = 0; // 0 = none, 1 = at least, 2 = at most, 3 = equal to.
	m_size = 0; // spin box content
	m_size_multiple = 1; // 0 = bytes, 1 = KB, 2 = MB
	m_modified_or_created = 0; // 0 = any, 1 = between, 2 = during the last
	m_date1 = QDate::currentDate();
	m_date2 = QDate::currentDate();
	m_date3_num = 0;
	m_modified_last = 0; // 0 = minutes, 1 = hours, 2 = days, 3 = months, 4 = years
	m_ignore_empty = true;
	m_size_comparison = true;
	m_md5_sum = true;
	m_binary_comparison = false;
	m_binary_comparison_size = 10; // in MB
	m_duplicates_right_url_only = false;
	m_empty_directories = false;
	m_include_subdirectories = true;
	m_include_hidden_items = true;
	m_use_locate = false;
	m_case_sensitive = true;
	m_soft_links = false;
	m_enable_duplicates_search = true;
	m_enable_missing_search = true;
	m_enable_newer_search = true;
	m_reference_directory_left = true;

	int i;
	for ( i=0; i<24; i++ ) m_filter_string.push_back( "" ); // name, path, owner, group from duplicates1, 2, missing1, 2, newer1, 2.
	m_filter_string.push_back( "EOL" );
	for ( i=0; i<62; i++ ) m_filter_int.push_back( 0 ); // All other entries. Bool is 1 or 0, Date is Year, Month, Day.

	// set correct default values now:
	// 0 and 1: min size, min size unit
	m_filter_int[2] = 9999; // duplicates1: 0..10
	m_filter_int[3] = 2;
	// 4: date check box
	m_filter_int[5] = QDate::currentDate().year();
	m_filter_int[6] = QDate::currentDate().month();
	m_filter_int[7] = QDate::currentDate().day();
	m_filter_int[8] = 1;
	m_filter_int[9] = 1;
	m_filter_int[10] = 1;

	m_filter_int[13] = 9999; // duplicates2: 11..19
	m_filter_int[14] = 2;
	m_filter_int[16] = QDate::currentDate().year();
	m_filter_int[17] = QDate::currentDate().month();
	m_filter_int[18] = QDate::currentDate().day();
	m_filter_int[19] = 1;

	m_filter_int[22] = 9999; // missing1: 20..28
	m_filter_int[23] = 2;
	m_filter_int[25] = QDate::currentDate().year();
	m_filter_int[26] = QDate::currentDate().month();
	m_filter_int[27] = QDate::currentDate().day();
	m_filter_int[28] = 1;

	m_filter_int[31] = 9999; // missing2: 29..37
	m_filter_int[32] = 2;
	m_filter_int[34] = QDate::currentDate().year();
	m_filter_int[35] = QDate::currentDate().month();
	m_filter_int[36] = QDate::currentDate().day();
	m_filter_int[37] = 1;

	m_filter_int[40] = 9999; // newer1: 38..49
	m_filter_int[41] = 2;
	m_filter_int[43] = QDate::currentDate().year();
	m_filter_int[44] = QDate::currentDate().month();
	m_filter_int[45] = QDate::currentDate().day();
	m_filter_int[46] = 1;
	m_filter_int[47] = 1;
	m_filter_int[48] = 1;
	m_filter_int[49] = 1;

	m_filter_int[52] = 9999; // newer2: 50..61
	m_filter_int[53] = 2;
	m_filter_int[55] = QDate::currentDate().year();
	m_filter_int[56] = QDate::currentDate().month();
	m_filter_int[57] = QDate::currentDate().day();
	m_filter_int[58] = 1;
	m_filter_int[59] = 1;
	m_filter_int[60] = 1;
	m_filter_int[61] = 1;

	m_virtual_subdirectories.clear();
}

QString KomparatorPreset::title()
{
	return m_preset_title;
}

bool KomparatorPreset::readConfig( uint preset_number )
{
	KConfigGroup cg( m_config, QString( "Preset%1" ).arg( preset_number ) );

	if ( !cg.exists() ) return false;

	// Read the entries
	m_preset_title = cg.readEntry( "preset_title" );
	m_url1 = KUrl( cg.readEntry( "url1" ) );
	m_url2 = KUrl( cg.readEntry( "url2" ) );
	m_regex_contains = cg.readEntry( "regex_contains" );
	m_regex_containsnot = cg.readEntry( "regex_containsnot" );
	m_size_selection = cg.readEntry( "size_selection", 0 ); // 0 = none, 1 = at least, 2 = at most, 3 = equal to.
	m_size = cg.readEntry( "size", 0 ); // spin box content
	m_size_multiple = cg.readEntry( "size_multiple", 0 ); // 0 = bytes, 1 = KB, 2 = MB
	m_modified_or_created = cg.readEntry( "modified_or_created", 0 ); // 0 = any, 1 = between, 2 = during the last
	m_date1 = cg.readEntry( "date1", QDate() );
	m_date2 = cg.readEntry( "date2", QDate() );
	m_date3_num = cg.readEntry( "date3_num", 0 );
	m_modified_last = cg.readEntry( "modified_last", 0 ); // 0 = minutes, 1 = hours, 2 = days, 3 = months, 4 = years
	m_ignore_empty = cg.readEntry( "ignore_empty", true );
	m_size_comparison = cg.readEntry( "size_comparison", true );
	m_md5_sum = cg.readEntry( "md5_sum", true );
	m_binary_comparison = cg.readEntry( "binary_comparison", false );
	m_binary_comparison_size = cg.readEntry( "binary_comparison_size", 3 ); // in MB
	m_duplicates_right_url_only = cg.readEntry( "duplicates_right_url_only", false );
	m_empty_directories = cg.readEntry( "empty_directories", 1 );
	m_include_subdirectories = cg.readEntry( "include_subdirectories", true );
	m_include_hidden_items = cg.readEntry( "include_hidden_items", false );
	m_use_locate = cg.readEntry( "use_locate", false );
	m_case_sensitive = cg.readEntry( "case_sensitive", true );
	m_soft_links = cg.readEntry( "soft_links", false );
	m_enable_duplicates_search = cg.readEntry( "enable_duplicates_search", true );
	m_enable_missing_search = cg.readEntry( "enable_missing_search", true );
	m_enable_newer_search = cg.readEntry( "enable_newer_search", true );
	m_reference_directory_left = cg.readEntry( "reference_directory_left", true );
	m_filter_string = cg.readEntry( "filter_string", QStringList() );
	m_filter_int = cg.readEntry( "filter_int", QList<int>() );
	m_virtual_subdirectories = cg.readEntry( "virtual_subdirectories", QStringList() );

	// Sanity check for values
	if ( m_regex_contains.isEmpty() ) m_regex_contains = "*";
	if ( m_size_selection > 3 ) m_size_selection = 0;
	if ( m_size_multiple > 2 ) m_size_multiple = 0;
	if ( m_modified_or_created > 2 ) m_modified_or_created = 0;
	if ( m_modified_last > 4 ) m_modified_last = 0;

	return true;
}

bool KomparatorPreset::saveConfig( uint preset_number )
{
	KConfigGroup cg( m_config, QString( "Preset%1" ).arg( preset_number ) );

	cg.writeEntry( "preset_title", m_preset_title );
	cg.writeEntry( "url1", m_url1.pathOrUrl() );
	cg.writeEntry( "url2", m_url2.pathOrUrl() );
	cg.writeEntry( "regex_contains", m_regex_contains );
	cg.writeEntry( "regex_containsnot", m_regex_containsnot );
	cg.writeEntry( "size_selection", m_size_selection );
	cg.writeEntry( "size", m_size );
	cg.writeEntry( "size_multiple", m_size_multiple );
	cg.writeEntry( "modified_or_created", m_modified_or_created );
	cg.writeEntry( "date1", m_date1 );
	cg.writeEntry( "date2", m_date2 );
	cg.writeEntry( "date3_num", m_date3_num );
	cg.writeEntry( "modified_last", m_modified_last );
	cg.writeEntry( "ignore_empty", m_ignore_empty );
	cg.writeEntry( "size_comparison", m_size_comparison );
	cg.writeEntry( "md5_sum", m_md5_sum );
	cg.writeEntry( "binary_comparison", m_binary_comparison );
	cg.writeEntry( "binary_comparison_size", m_binary_comparison_size );
	cg.writeEntry( "duplicates_right_url_only", m_duplicates_right_url_only );
	cg.writeEntry( "empty_directories", m_empty_directories );
	cg.writeEntry( "include_subdirectories", m_include_subdirectories );
	cg.writeEntry( "include_hidden_items", m_include_hidden_items );
	cg.writeEntry( "use_locate", m_use_locate );
	cg.writeEntry( "case_sensitive", m_case_sensitive );
	cg.writeEntry( "soft_links", m_soft_links );
	cg.writeEntry( "enable_duplicates_search", m_enable_duplicates_search );
	cg.writeEntry( "enable_missing_search", m_enable_missing_search );
	cg.writeEntry( "enable_newer_search", m_enable_newer_search );
	cg.writeEntry( "reference_directory_left", m_reference_directory_left );
	cg.writeEntry( "filter_string", m_filter_string );
	cg.writeEntry( "filter_int", m_filter_int );
	cg.writeEntry( "virtual_subdirectories", m_virtual_subdirectories );

	cg.sync();

	return true;
}

bool KomparatorPreset::readFromInterface( bool reread )
{
	KDialog *ask_name_dialog = new KDialog( (QWidget *)m_parent );
	ask_name_dialog->setCaption( i18n( "Add preset" ) );
	ask_name_dialog->setButtons( KDialog::Ok | KDialog::Cancel );

	QWidget *page = new QWidget( ask_name_dialog );
	QVBoxLayout *page_layout = new QVBoxLayout( page );
	page->setLayout( page_layout );

	QLabel *preset_label = new QLabel( i18n( "Name of your new preset:" ), (QWidget*)page );
	page_layout->addWidget( preset_label );
	QLineEdit *preset_title_LineEdit = new QLineEdit( "", (QWidget*)page );
	page_layout->addWidget( preset_title_LineEdit );
	if ( reread )
	{
		preset_label->setText( i18n( "Name of your preset:" ) );
		preset_title_LineEdit->setReadOnly( true );
		preset_title_LineEdit->setText( title() );
	}
	preset_title_LineEdit->setMinimumWidth( ask_name_dialog->fontMetrics().maxWidth()*20 );

	QLabel *url1_label = new QLabel( i18n( "Left URL:" ), (QWidget*)page );
	page_layout->addWidget( url1_label );
	KUrl url_show = KUrl( m_parent->url1_KUrlRequester->url() ).prettyUrl( KUrl::RemoveTrailingSlash ); // no such function as "pathOrUrl( int trailing )"!
	QLineEdit *url1_LineEdit = new QLineEdit( url_show.pathOrUrl(), (QWidget*)page );
	page_layout->addWidget( url1_LineEdit );
	url1_LineEdit->setMinimumWidth( ask_name_dialog->fontMetrics().maxWidth()*20 );
	if ( reread )
	{
		url1_label->setText( url1_label->text() + " (" + i18n( "Old URL:" ) + " \"" + m_url1.pathOrUrl() + "\" )" );
	}

	QLabel *url2_label = new QLabel( i18n( "Right URL:" ), (QWidget*)page );
	page_layout->addWidget( url2_label );
	url_show = KUrl( m_parent->url2_KUrlRequester->url() ).prettyUrl( KUrl::RemoveTrailingSlash ); // no such function as "pathOrUrl( int trailing )"!
	QLineEdit *url2_LineEdit = new QLineEdit( url_show.pathOrUrl(), (QWidget*)page );
	page_layout->addWidget( url2_LineEdit );
	url2_LineEdit->setMinimumWidth( ask_name_dialog->fontMetrics().maxWidth()*20 );
	if ( reread )
	{
		url2_label->setText( url2_label->text() + " (" + i18n( "Old URL:" ) + " \"" + m_url2.pathOrUrl() + "\" )" );
	}

	ask_name_dialog->setMainWidget( page );

	if ( ask_name_dialog->exec() == QDialog::Rejected )
	{
		delete ask_name_dialog;
		ask_name_dialog = NULL;
		return false;
	}

	m_preset_title = preset_title_LineEdit->text();
	m_url1 = KUrl( url1_LineEdit->text() );
	m_url2 = KUrl( url2_LineEdit->text() );

	delete ask_name_dialog;
	ask_name_dialog = NULL;

	// Set entries from interface
	m_regex_contains = m_parent->regexContains_KLineEdit->text();
	m_regex_containsnot = m_parent->regexContainsNot_KLineEdit->text();
	m_size_selection = m_parent->size_QComboBox->currentIndex(); // 0 = none, 1 = at least, 2 = at most, 3 = equal to.
	m_size = m_parent->bytes_KIntNumInput->value(); // spin box content
	m_size_multiple = m_parent->bytes_QComboBox->currentIndex(); // 0 = bytes, 1 = KB, 2 = MB
	m_modified_or_created = m_parent->modifiedNoRestriction_RadioButton->isChecked() ? 0 :
												( m_parent->modifiedBetween_RadioButton->isChecked() ? 1 : 2 ); // 0 = any, 1 = between, 2 = during the last
	QDate *tmp_date = new QDate;
	m_date1 = m_parent->modifiedFrom_KDateCombo->getDate( tmp_date );
	m_date2 = m_parent->modifiedTo_KDateCombo->getDate( tmp_date );
	m_date3_num = m_parent->modifiedLastTime_KIntNumInput->value();
	m_modified_last = m_parent->modifiedLastTime_QComboBox->currentIndex(); // 0 = minutes, 1 = hours, 2 = days, 3 = months, 4 = years
	m_ignore_empty = m_parent->ignoreEmptyFiles_CheckBox->isChecked();
	m_size_comparison = m_parent->size_CheckBox->isChecked();
	m_md5_sum = m_parent->calculateChecksum_CheckBox->isChecked();
	m_binary_comparison = m_parent->binaryComparison_CheckBox->isChecked();
	m_binary_comparison_size = m_parent->binaryComparison_KIntSpinBox->value(); // in MB
	m_duplicates_right_url_only = m_parent->duplicatesOnly_CheckBox->isChecked();
	m_empty_directories = m_parent->findEmptyDirectories_CheckBox->isChecked();
	m_include_subdirectories = m_parent->includeSubdirectories_CheckBox->isChecked();
	m_include_hidden_items = m_parent->includeHiddenItems_CheckBox->isChecked();
	m_use_locate = m_parent->useFilesIndex_CheckBox->isChecked();
	m_case_sensitive = m_parent->caseSensitiveSearch_CheckBox->isChecked();
	m_soft_links = m_parent->followSoftLinks_CheckBox->isChecked();
	m_enable_duplicates_search = m_parent->isEnabled( DUPLICATES );
	m_enable_missing_search = m_parent->isEnabled( MISSING );
	m_enable_newer_search = m_parent->isEnabled( NEWER );
	m_reference_directory_left = m_parent->duplicatesReference1_RadioButton->isChecked();

	QStringList string_list;
	string_list = m_parent->duplicates1_KomparatorFileFilter->getStringList() + m_parent->duplicates2_KomparatorFileFilter->getStringList() +
		m_parent->missing1_KomparatorFileFilter->getStringList() + m_parent->missing2_KomparatorFileFilter->getStringList() +
		m_parent->newer1_KomparatorFileFilter->getStringList() + m_parent->newer2_KomparatorFileFilter->getStringList();
	string_list.push_back( "EOL" );
	m_filter_string = string_list;

	QList<int> int_list;
	int_list = m_parent->duplicates1_KomparatorFileFilter->getIntList() + m_parent->duplicates2_KomparatorFileFilter->getIntList() +
		m_parent->missing1_KomparatorFileFilter->getIntList() + m_parent->missing2_KomparatorFileFilter->getIntList() +
		m_parent->newer1_KomparatorFileFilter->getIntList() + m_parent->newer2_KomparatorFileFilter->getIntList();
	m_filter_int = int_list;

	KUrl::List virtual_subdirectories = m_parent->subdirectories_KomparatorUrlList->getUrlList();
	m_virtual_subdirectories.clear();
	KUrl::List::Iterator it;
	for ( it=virtual_subdirectories.begin(); it != virtual_subdirectories.end(); ++it )
	{
		m_virtual_subdirectories.push_back( (*it).url() );
	}

	return true;
}

void KomparatorPreset::setToInterface()
{
	// Set the entries in the GUI
	if ( !m_url1.isEmpty() ) m_parent->url1_KUrlRequester->setUrl( m_url1 );
	if ( !m_url2.isEmpty() ) m_parent->url2_KUrlRequester->setUrl( m_url2 );
	m_parent->regexContains_KLineEdit->setText( m_regex_contains );
	m_parent->regexContainsNot_KLineEdit->setText( m_regex_containsnot );
	m_parent->size_QComboBox->setCurrentIndex( m_size_selection );
	m_parent->bytes_KIntNumInput->setValue( m_size );
	m_parent->bytes_QComboBox->setCurrentIndex( m_size_multiple );
	m_parent->modifiedNoRestriction_RadioButton->setChecked( ( m_modified_or_created == 0 ) ? true : false );
	m_parent->modifiedBetween_RadioButton->setChecked( ( m_modified_or_created == 1 ) ? true : false );
	m_parent->modifiedLastTime_RadioButton->setChecked( ( m_modified_or_created == 2 ) ? true : false );
	m_parent->modifiedFrom_KDateCombo->setDate( m_date1 );
	m_parent->modifiedTo_KDateCombo->setDate( m_date2 );
	m_parent->modifiedLastTime_KIntNumInput->setValue( m_date3_num );
	m_parent->modifiedLastTime_QComboBox->setCurrentIndex( m_modified_last );
	m_parent->ignoreEmptyFiles_CheckBox->setChecked( m_ignore_empty );
	m_parent->size_CheckBox->setChecked( m_size_comparison );
	m_parent->calculateChecksum_CheckBox->setChecked( m_md5_sum );
	m_parent->binaryComparison_CheckBox->setChecked( m_binary_comparison );
	m_parent->binaryComparison_KIntSpinBox->setValue( m_binary_comparison_size );
	m_parent->duplicatesOnly_CheckBox->setChecked( m_duplicates_right_url_only );
	m_parent->findEmptyDirectories_CheckBox->setChecked( m_empty_directories );
	m_parent->includeSubdirectories_CheckBox->setChecked( m_include_subdirectories );
	m_parent->includeHiddenItems_CheckBox->setChecked( m_include_hidden_items );
	m_parent->useFilesIndex_CheckBox->setChecked( m_use_locate );
	m_parent->caseSensitiveSearch_CheckBox->setChecked( m_case_sensitive );
	m_parent->followSoftLinks_CheckBox->setChecked( m_soft_links );
	m_parent->main_TabWidget->setTabIcon( 0,
		KIconLoader::global()->loadIcon( m_enable_duplicates_search ? "opinion-okay" : "opinion-no", KIconLoader::Toolbar, KIconLoader::SizeSmall ) );
	m_parent->main_TabWidget->setTabIcon( 1,
		KIconLoader::global()->loadIcon( m_enable_missing_search ? "opinion-okay" : "opinion-no", KIconLoader::Toolbar, KIconLoader::SizeSmall ) );
	m_parent->main_TabWidget->setTabIcon( 2,
		KIconLoader::global()->loadIcon( m_enable_newer_search ? "opinion-okay" : "opinion-no", KIconLoader::Toolbar, KIconLoader::SizeSmall ) );
	m_parent->duplicatesReference1_RadioButton->setChecked( m_reference_directory_left );
	m_parent->duplicatesReference2_RadioButton->setChecked( !m_reference_directory_left );

	if ( m_filter_string.size() == 25 && m_filter_string[24] == "EOL" )
	{
		QStringList string_list;
		string_list << m_filter_string[0] << m_filter_string[1] << m_filter_string[2] << m_filter_string[3];
		m_parent->duplicates1_KomparatorFileFilter->setStringList( string_list );
		string_list.clear();
		string_list << m_filter_string[4] << m_filter_string[5] << m_filter_string[6] << m_filter_string[7];
		m_parent->duplicates2_KomparatorFileFilter->setStringList( string_list );
		string_list.clear();
		string_list << m_filter_string[8] << m_filter_string[9] << m_filter_string[10] << m_filter_string[11];
		m_parent->missing1_KomparatorFileFilter->setStringList( string_list );
		string_list.clear();
		string_list << m_filter_string[12] << m_filter_string[13] << m_filter_string[14] << m_filter_string[15];
		m_parent->missing2_KomparatorFileFilter->setStringList( string_list );
		string_list.clear();
		string_list << m_filter_string[16] << m_filter_string[17] << m_filter_string[18] << m_filter_string[19];
		m_parent->newer1_KomparatorFileFilter->setStringList( string_list );
		string_list.clear();
		string_list << m_filter_string[20] << m_filter_string[21] << m_filter_string[22] << m_filter_string[23];
		m_parent->newer2_KomparatorFileFilter->setStringList( string_list );
	}

	if ( m_filter_int.size() == 62 )
	{
		QList<int> int_list;
		int_list << m_filter_int[0] << m_filter_int[1] << m_filter_int[2] << m_filter_int[3] << m_filter_int[4] <<
			m_filter_int[5] << m_filter_int[6] << m_filter_int[7] << m_filter_int[8] << m_filter_int[9] << m_filter_int[10];
		m_parent->duplicates1_KomparatorFileFilter->setIntList( int_list );
		int_list.clear();
		int_list << m_filter_int[11] << m_filter_int[12] << m_filter_int[13] << m_filter_int[14] << m_filter_int[15] <<
			m_filter_int[16] << m_filter_int[17] << m_filter_int[18] << m_filter_int[19];
		m_parent->duplicates2_KomparatorFileFilter->setIntList( int_list );
		int_list.clear();
		int_list << m_filter_int[20] << m_filter_int[21] << m_filter_int[22] << m_filter_int[23] << m_filter_int[24] <<
			m_filter_int[25] << m_filter_int[26] << m_filter_int[27] << m_filter_int[28];
		m_parent->missing1_KomparatorFileFilter->setIntList( int_list );
		int_list.clear();
		int_list << m_filter_int[29] << m_filter_int[30] << m_filter_int[31] << m_filter_int[32] << m_filter_int[33] <<
			m_filter_int[34] << m_filter_int[35] << m_filter_int[36] << m_filter_int[37];
		m_parent->missing2_KomparatorFileFilter->setIntList( int_list );
		int_list.clear();
		int_list << m_filter_int[38] << m_filter_int[39] << m_filter_int[40] << m_filter_int[41] << m_filter_int[42] <<
			m_filter_int[43] << m_filter_int[44] << m_filter_int[45] << m_filter_int[46] << m_filter_int[47] <<
				m_filter_int[48] << m_filter_int[49];
		m_parent->newer1_KomparatorFileFilter->setIntList( int_list );
		int_list.clear();
		int_list << m_filter_int[50] << m_filter_int[51] << m_filter_int[52] << m_filter_int[53] << m_filter_int[54] <<
			m_filter_int[55] << m_filter_int[56] << m_filter_int[57] << m_filter_int[58] << m_filter_int[59] <<
				m_filter_int[60] << m_filter_int[61];
		m_parent->newer2_KomparatorFileFilter->setIntList( int_list );
	}

	m_parent->subdirectories_KomparatorUrlList->clear();
	QStringList::Iterator it;
	for ( it=m_virtual_subdirectories.begin(); it!=m_virtual_subdirectories.end(); ++it )
	{
		m_parent->subdirectories_KomparatorUrlList->addUrl( KUrl( *it ) );
	}
}
