/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file pixel.hpp
 * \brief Representation of a pixel in image processing.
 * \author Julien Jorge
 */
#ifndef __CLAW_PIXEL_HPP_
#define __CLAW_PIXEL_HPP_

namespace claw
{
  namespace graphic
  {
    /**
     * \brief RGB pixel.
     */
    struct pixel24
    {
      typedef unsigned char component_type;

      /** \brief Component by component representation. */
      struct
      {
        /** \brief Red component. */
        component_type red;

        /** \brief Green component. */
        component_type green;

        /** \brief Blue component. */
        component_type blue;

      } components; 

    public:
      /** \brief Default constructor. */
      pixel24() {}

      /**
       * \brief Constructor.
       * \param r The value of the red field.
       * \param g The value of the green field.
       * \param b The value of the blue field.
       */
      pixel24( component_type r, component_type g, component_type b )
      { 
        components.red = r;
        components.green = g;
        components.blue = b;
      } // pixel24()

    }; // struct pixel24

    /**
     * \brief RGBA pixel.
     */
    union pixel32
    {
      typedef unsigned char component_type;

      /** \brief Compressed representation. */
      unsigned int pixel;

      /** \brief Component by component representation. */
      struct
      {
        /** \brief Red component. */
        component_type red;

        /** \brief Green component. */
        component_type green;

        /** \brief Blue component. */
        component_type blue;

        /** \brief Translucy. */
        component_type alpha;

      } components; 

    public:
      /** \brief Default constructor. */
      pixel32() {}

      /**
       * \brief Constructor.
       * \param r The value of the red field.
       * \param g The value of the green field.
       * \param b The value of the blue field.
       * \param a The value of the alpha field.
       */
      pixel32( component_type r, component_type g, component_type b,
	       component_type a )
      { 
        components.red = r;
        components.green = g;
        components.blue = b;
        components.alpha = a;
      } // pixel32()

      /**
       * \brief Assignement operator.
       * \param that The other pixel.
       * \remark The alpha component of the \a this is set to 255.
       */
      pixel32& operator=( const pixel24& that )
      {
        components.red = that.components.red;
        components.green = that.components.green;
        components.blue = that.components.blue;
        components.alpha = 255;

        return *this;
      } // operator=()

      /**
       * \brief Tell if two pixels are equal.
       * \param that The other pixel.
       */
      bool operator==( const pixel32& that ) const
      {
        return pixel == that.pixel;
      } // operator==()

      /**
       * \brief Tell if two pixels are different.
       * \param that The other pixel.
       */
      bool operator!=( const pixel32& that ) const
      {
        return pixel != that.pixel;
      } // operator!=()

      /**
       * \brief Get the luminosity of the pixel.
       * \return The luminosity between 0 (dark) and 255 (light).
       *
       * The luminosity is approximated with integers:
       * \f$0,715160 \simeq 183 \div 256 = 0,71484375 \f$
       * \f$0,212671 \simeq 54 \div 256 = 0,2109375 \f$
       * \f$0,072169 \simeq 18 \div 256 = 0,0703125 \f$
       */
      component_type luminosity() const
      {
        return ((unsigned int)components.red * 183 
                + (unsigned int)components.green * 54 
                + (unsigned int)components.blue * 18
                ) / 256;
      } // luminosity()

    };
  } // namespace graphic
} // namespace claw

#endif // __CLAW_PIXEL_HPP__
