/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.logging.jdk;

import java.util.logging.Logger;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import org.jboss.logging.LoggerPlugin;
import org.jboss.logging.MDCProvider;
import org.jboss.logging.MDCSupport;
import org.jboss.logging.NDCProvider;
import org.jboss.logging.NDCSupport;

/** An example LoggerPlugin which uses the JDK java.util.logging framework.
 *
 * @author Scott.Stark@jboss.org
 * @version $Revison:$
 */
public class JDK14LoggerPlugin implements LoggerPlugin, MDCSupport, NDCSupport
{
   private Logger log;
   private String name;

   public void init(String name)
   {
      this.name = name;
      log = Logger.getLogger(name);
   }

   private void doLog(Level level, Object message, Throwable t) {
      LogRecord record = new LogRecord(level, message.toString());
      record.setLoggerName(name);
      record.setThrown(t);
      record.setSourceMethodName(null); // prevent expensive, yet pointless, lookup
      log.log(record);
   }

   public boolean isTraceEnabled()
   {
      return log.isLoggable(JBossLevel.TRACE);
   }

   public void trace(Object message)
   {
      doLog(JBossLevel.TRACE, message, null);
   }

   public void trace(Object message, Throwable t)
   {
      doLog(JBossLevel.TRACE, message, t);
   }

   public boolean isDebugEnabled()
   {
      return log.isLoggable(JBossLevel.DEBUG);
   }

   public void debug(Object message)
   {
      doLog(JBossLevel.DEBUG, message, null);
   }

   public void debug(Object message, Throwable t)
   {
      doLog(JBossLevel.DEBUG, message, t);
   }

   public boolean isInfoEnabled()
   {
      return log.isLoggable(JBossLevel.INFO);
   }

   public void info(Object message)
   {
      doLog(JBossLevel.INFO, message, null);
   }

   public void info(Object message, Throwable t)
   {
      doLog(JBossLevel.INFO, message, t);
   }

   public void warn(Object message)
   {
      doLog(JBossLevel.WARN, message, null);
   }

   public void warn(Object message, Throwable t)
   {
      doLog(JBossLevel.WARN, message, t);
   }

   public void error(Object message)
   {
      doLog(JBossLevel.ERROR, message, null);
   }

   public void error(Object message, Throwable t)
   {
      doLog(JBossLevel.ERROR, message, t);
   }

   public void fatal(Object message)
   {
      doLog(JBossLevel.FATAL, message, null);
   }

   public void fatal(Object message, Throwable t)
   {
      doLog(JBossLevel.FATAL, message, t);
   }

   public NDCProvider getNDCProvider()
   {
      return new JDKNDCProvider();
   }

   public MDCProvider getMDCProvider()
   {
      return new JDKMDCProvider();
   }
}
