/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <stdlib.h>
#include <sys/devicestat.h>
#include <stdio.h>

#include "libsysactivity.h"

#define	DEVSTAT_VERSION_6_SUPPORTED 6
#define MIB_SIZE 3

static int read_devs(int* number_devs, struct devstat** devs);
static void assign(struct sa_data_storage* dst, struct devstat* src);

__thread void* buffer;
__thread size_t buffer_size = 0;
__thread int ds_mib[MIB_SIZE];

int sa_open_data_storage() {
	buffer = NULL;
	buffer_size = 0;

	int version;
	size_t len = sizeof version;
	if (sysctlbyname("kern.devstat.version", &version, &len, NULL, 0) == -1)
		return ENOSYS;

	if (version != DEVSTAT_VERSION_6_SUPPORTED)
		return ENOSYS;

	len = MIB_SIZE;
	if (sysctlnametomib("kern.devstat.numdevs", ds_mib, &len) == -1)
		return ENOSYS;

	return 0;
}

int sa_close_data_storage() {
	if (buffer != NULL)
		free(buffer);
	return 0;
}

int sa_count_data_storages(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	int number_devs;
	size_t len = sizeof number_devs;
	if (sysctl(ds_mib, MIB_SIZE, &number_devs, &len, NULL, 0) == -1)
		return ENOSYS;

	*number = number_devs;
	return 0;
}

int sa_get_data_storage(char* name, struct sa_data_storage* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	int number_devs;
	struct devstat* devices;
	int ret = read_devs(&number_devs, &devices);
	if (ret != 0)
		return ret;

	uint16_t i;
	char tmp_name[sizeof dst->name];
	for (i = 0; i < number_devs; i++) {
		snprintf(tmp_name, sizeof tmp_name, "%s%d", devices[i].device_name, devices[i].unit_number);
		if (strncmp(tmp_name, name, sizeof tmp_name) != 0)
			continue;

		assign(dst, &devices[i]);
		return 0;
	}
	return ENODEV;
}

int sa_get_data_storages(struct sa_data_storage* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	int number_devs;
	struct devstat* devices;
	int ret = read_devs(&number_devs, &devices);
	if (ret != 0)
		return ret;

	*written = 0;
	uint16_t i;
	for (i = 0; i < dst_size; i++) {
		if (i == dst_size)
			return ENOMEM;
		if (i == number_devs)
			return ENODEV;

		assign(&dst[i], &devices[i]);
		(*written)++;
	}
	return 0;
}

static int read_devs(int* number_devs, struct devstat** devs) {
	size_t len = sizeof *number_devs;
	if (sysctlbyname("kern.devstat.numdevs", number_devs, &len, NULL, 0) == -1)
		return ENOSYS;

	len = (*number_devs * sizeof(struct devstat)) + sizeof(long);
	if (len > buffer_size) {
		buffer = realloc(buffer, len);
		if (buffer == NULL)
			return ENOMEM;
		buffer_size = len;
	}

	if (sysctlbyname("kern.devstat.all", buffer, &len, NULL, 0) == -1)
		return ENOSYS;
	*devs = (struct devstat *) (buffer + sizeof(long));
	return 0;
}

static void assign(struct sa_data_storage* dst, struct devstat* src) {
	snprintf(dst->name, sizeof dst->name, "%s%d", src->device_name, src->unit_number);
	dst->reads = src->operations[DEVSTAT_READ];
	dst->time_spent_reading = src->duration[DEVSTAT_READ].sec;
	dst->writes = src->operations[DEVSTAT_WRITE];
	dst->time_spent_writing = src->duration[DEVSTAT_WRITE].sec;
	dst->bytes_read = src->bytes[DEVSTAT_READ];
	dst->bytes_written = src->bytes[DEVSTAT_WRITE];
}
