/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <inttypes.h>
#include <kstat.h>
#include <string.h>

static kstat_t* get_next_ksp(kstat_t* local_ksp);
static int assign_dst(struct sa_net_interface* dst, kstat_t* local_ksp) SA_NONNULL;
static void assign_field(kstat_t* local_ksp, uint64_t* field, char* tag64, char* tag32) SA_NONNULL SA_HOT;

__thread kstat_ctl_t* network_kc;
__thread kstat_t* ksp;

int sa_open_net() {
	network_kc = kstat_open();
	if (network_kc == NULL)
		return ENOSYS;

	return 0;
}

int sa_close_net() {
	if (network_kc != NULL)
		kstat_close(network_kc);

	return 0;
}

int sa_count_net_interfaces(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	kstat_t* local_ksp = NULL;
	*number = 0;
	while ((local_ksp = get_next_ksp(local_ksp)) != NULL)
		++(*number);

	return 0;
}

int sa_reset_net_interfaces() {
	ksp = NULL;
	return 0;
}

int sa_get_net_interfaces_ids(char* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	int i;
	*written = 0;
	for (i = 0; (ksp = get_next_ksp(ksp)) != NULL; ++i) {
		if (i == dst_size)
			return ENOMEM;

		if (kstat_read(network_kc, ksp, NULL) == -1)
			return 0;
		strncpy(&dst[i * SA_NET_INTERFACE_NAME], ksp->ks_name, SA_NET_INTERFACE_NAME);
		++(*written);
	}
	return 0;
}

int sa_get_net_interface(char* name, struct sa_net_interface* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	kstat_t* local_ksp = NULL;
	while ((local_ksp = get_next_ksp(local_ksp)) != NULL) {
		if (strncmp(local_ksp->ks_name, name, sizeof local_ksp->ks_name) != 0)
			continue;

		if (!assign_dst(dst, local_ksp))
			return ENOSYS;
		return 0;
	}
	return ENODEV;
}

int sa_get_net_interfaces(struct sa_net_interface* dst, uint16_t dst_size,
		uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	int i;
	*written = 0;
	for (i = 0; (ksp = get_next_ksp(ksp)) != NULL; ++i) {
		if (i == dst_size)
			return ENOMEM;

		if (!assign_dst(&dst[i], ksp))
			return ENOSYS;
		++(*written);
	}
	return 0;
}

static kstat_t* get_next_ksp(kstat_t* local_ksp) {
	do {
		local_ksp = (local_ksp == NULL) ? network_kc->kc_chain : local_ksp->ks_next;

		if (local_ksp == NULL)
			return local_ksp;
	} while (strcmp(local_ksp->ks_module, "link") != 0 && strcmp(local_ksp->ks_module, "lo") != 0);

	return local_ksp;
}

static int assign_dst(struct sa_net_interface* dst, kstat_t* local_ksp) {
	if (kstat_read(network_kc, local_ksp, NULL) == -1)
		return 0;

	strncpy(dst->name, local_ksp->ks_name, sizeof dst->name);
	dst->received_bytes = 0;
	assign_field(local_ksp, &dst->received_bytes, "rbytes64", "rbytes");
	assign_field(local_ksp, &dst->received_packets, "ipackets64", "ipackets");
	kstat_named_t* knp = kstat_data_lookup(local_ksp, "ierrors");
	dst->received_errors = (knp != NULL) ? knp->value.ui32 : 0;

	assign_field(local_ksp, &dst->sent_bytes, "obytes64", "obytes");
	assign_field(local_ksp, &dst->sent_packets, "opackets64", "opackets");
	knp = kstat_data_lookup(local_ksp, "oerrors");
	dst->sent_errors = (knp != NULL) ? knp->value.ui32 : 0;
	return 1;
}

static void assign_field(kstat_t* local_ksp, uint64_t* field, char* tag64, char* tag32) {
	kstat_named_t* knp = kstat_data_lookup(local_ksp, tag64);
	if (knp != NULL) {
		*field = knp->value.ui64;
		return;
	}

	knp = kstat_data_lookup(local_ksp, tag32);
	*field = (knp != NULL) ? knp->value.ui32 : 0;
}
