/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file VisualAction.h
    \brief Implementation of the Template class ActionNode.
    
    Magics Team - ECMWF 2007
    
    Started: Tue 6-Mar-2007
    
    Changes:
    
*/ 


#include "LegendVisitor.h"
#include "Timer.h"
#include "TextVisitor.h"
#include "SceneVisitor.h"
#include "AnimationRules.h"
#include "Layer.h"
#include "MagnifierVisitor.h"


using namespace magics;

template <class P>
VisualAction<P>::VisualAction() : data_(0)
{
}

template <class P>
VisualAction<P>::~VisualAction() 
{
	if (data_) delete data_;
}

/*!
 Class information are given to the output-stream.
*/	
template <class P>	
void VisualAction<P>::print(ostream& out)  const
{
	out << "ActionNode<P>";
}

template <class P>	
void VisualAction<P>::getReady()
{
	Log::dev() << "ActionNode<P>::getReady()\n";
}

template <class P>	
void VisualAction<P>::visit(DrawingVisitor& drawing)
{

	if ( !data_ || visdefs_.empty() )
	{
		Log::warning() << " No proper action  defined " << endl;
		return;
	}
	data_->getReady(drawing.transformation());

	for ( typename vector<Visdef<P>* >::iterator visdef = visdefs_.begin(); visdef != visdefs_.end(); ++visdef)
	{
		Timer timer("plotting", "time spent in plotting");
		(**visdef)(*data_, drawing.layout()); // Now the visualObject ahs the responsability to reproject!
	}
}

template <class P>	
void VisualAction<P>::visit(HistoVisitor& drawing)
{
	cout << "GREAT ---> BUILD the Histogram!!" << endl;
	for ( typename vector<Visdef<P>* >::iterator visdef = visdefs_.begin(); visdef != visdefs_.end(); ++visdef)
		(**visdef).visit(*data_, drawing);
}



template <class P>	
void VisualAction<P>::visit(Transformation& transformation)
{
	if ( !data_ || visdefs_.empty() ) {
		Log::warning() << " No proper action  defined " << endl;
		return;
	}
	data_->visit(transformation);	
	for ( typename vector<Visdef<P>* >::iterator visdef = visdefs_.begin(); visdef != visdefs_.end(); ++visdef)
		(**visdef).visit(transformation, *data_);
}
 
 
template <class P>	
void VisualAction<P>::visit(LegendVisitor& legend)
{
	bool needEmpty = false;
	for ( typename vector<Visdef<P>* >::iterator visdef = visdefs_.begin(); visdef != visdefs_.end(); ++visdef) {
		if ( needEmpty ) 
			legend.add(new EmptyEntry() );
		else 
			needEmpty = true;
		(**visdef).visit(legend);
	}
	data_->visit(legend);
}

template <class P>	 
void VisualAction<P>::visit(TextVisitor& title)
{
	if ( data_ ) data_->visit(title);
}

template <class P>	 
void VisualAction<P>::visit(MetaDataVisitor& infos)
{
	if ( data_ ) data_->visit(infos);
	for ( typename vector<Visdef<P>* >::iterator visdef = this->visdefs_.begin(); visdef != this->visdefs_.end(); ++visdef) {
		(*visdef)->visit(infos);
	}
}

template <class P>	 
void VisualAction<P>::visit(MetaDataCollector& infos)
{
	if ( data_ ) data_->visit(infos);
}

template <class P>	 
void VisualAction<P>::visit(ValuesCollector& infos)
{
	infos.transformation(&transformation());
	if ( data_ ) data_->visit(infos);
}

template <class P>	 
void VisualAction<P>::visit(MagnifierCollector& infos)
{
	infos.transformation(&transformation());
	if ( data_ ) data_->visit(infos);
}

template <class P>	 
void VisualAction<P>::visit(AnimationRules& rules)
{
	if ( data_ ) data_->visit(rules);
}

template <class P>	 
const string& VisualAction<P>::name()
{
	static string name = "unknown";
	return ( data_ ) ? data_->name() : name;
}

template <class P>	 
const string& VisualAction<P>::id() {
	static string name = "unknown";
	return ( data_ ) ? data_->layerId() : name;
}


	
template <class P>	
VisualAnimation<P>::VisualAnimation():loop_(0)
{
}

template <class P>	
VisualAnimation<P>::~VisualAnimation()
{
	if (loop_) delete loop_;
}

template <class P>
void VisualAnimation<P>::prepare() 
{
	if ( !this->empty() )
		return;
	
	layer_ = new StepLayer();
	layer_->name(loop_->name());
	layer_->id(loop_->name()); 
	loop_->visit(*layer_);
	
	loop_->setToFirst(); 
	while ( loop_->hasMore() )
	{
		Log::dev() << "New Frame" << endl; 

		VisualAction<P>* action = new VisualAction<P>();
		action->parent(this);
		action->data(loop_->current());

		for ( typename vector<Visdef<P>* >::iterator visdef = this->visdefs_.begin(); visdef != this->visdefs_.end(); ++visdef) {    
			action->visdef(*visdef);				
		}

		layer_->addStep(action);

		//To put infos from data to layer!
		//loop_->current()->visit(**(layer_->endStep()));

		loop_->next();
	}
}


template <class P>	
void VisualAnimation<P>::visit(MetaDataVisitor&)
{
}

template <class P>	
void VisualAnimation<P>::visit(AnimationRules& rules)
{
	prepare();
	for ( iterator entry = this->begin(); entry != this->end(); ++entry)
		(*entry)->visit(rules);
	rules.add(*this->layer_);
}


template <class P>	
void VisualAnimation<P>::visit(Transformation& transformation)
{
	assert(loop_);
	loop_->visit(transformation);
} 

template <class P>	
void VisualAnimation<P>::visit(SceneLayer& layer, vector<LayoutVisitor*>& visitors)
{
   	layer.add(layer_);
 	for  (vector<LayoutVisitor*>::iterator visitor = visitors.begin(); visitor != visitors.end(); ++visitor) {
   		layer.addVisitor(*visitor); 
   		layer_->addVisitor(*visitor);  
   	}
}

template <class P>	
void VisualAction<P>::visit(SceneLayer& layer, vector<LayoutVisitor*>& visitors)
{
	StaticLayer* plot = new StaticLayer(this);
	
	//Do we need to call visit here, or it is enough to call it later (se  below)
	if ( data_ ) 
		data_->visit(*plot);
 	layer.add(plot);

	for  (vector<LayoutVisitor*>::iterator visitor = visitors.begin(); visitor != visitors.end(); ++visitor) {
		plot->set(*visitor);
		(*visitor)->visit(*this);
	}

	//Is it safe to call visit here?
	//if ( data_ ) 
	//	data_->visit(*plot);
}
