/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file MgQLayerNode.cc
    \brief Implementation of the MgQLayerNode class.
    \author Graphics Section, ECMWF

    Started: February 2010
*/

#include <QDebug>
#include <QImage>
#include <QPainter>

#include "MgQLayerNode.h"

#include "MgQHistoNode.h"
#include "MgQLayoutNode.h"
#include "MgQScene.h"
#include "MgQStepNode.h"

using namespace magics;

MgQLayerNode::MgQLayerNode(QString name,Layer &layer,MgQLayoutNode *layout,int stepNum) : 
	   MgQBaseNode(name), layer_(layer), layout_(layout)
{
	rootNode_=new MgQBaseNode("LayerRootNode");
	rootNode_->setParentItem(this);
	rootNodeIsChild_=true;

	magnifierLayout_=0;
	magnifierDataEnabled_=false;

	alphaScene_=0;
	previewScene_=0;
	histoScene_=0;
	//rootNode_->setTransform(layout->sceneTransform());

	for(int i=0; i < stepNum; i++)
	{
		steps_ << 0;
	}

	for(int i=0; i < stepNum; i++)
	{
		histoNodes_ << 0;
	}
	if(stepNum == 0)
	{
		histoNodes_ << 0;
	}

}

MgQMagnifierLayoutNode * MgQLayerNode::magnifierLayoutNode()
{
	if(!magnifierLayout_)//Init magnifier layout node
	{
		MgQLayoutNode *topLayout=topLayoutNode();
		if(!topLayout)
			return 0;

		//it is outside the root node
		magnifierLayout_= new MgQMagnifierLayoutNode(topLayout);
		magnifierLayout_->setParentItem(this);
		magnifierLayout_->setParentItemInMainScene(this);
		magnifierLayout_->setData(MgQ::NodeTypeKey,MgQ::LayoutNode);
		magnifierLayout_->setData(MgQ::NodeIsVisibleKey,false);
	}
	
	return magnifierLayout_;
}


bool MgQLayerNode::layerVisibility() const
{
	return layer_.visibility();
}

void MgQLayerNode::setLayerVisibility(bool visible)
{	
	layer_.visibility(visible);
	//setVisible(b);
	setData(MgQ::NodeIsVisibleKey,visible);
}

float MgQLayerNode::layerAlpha() const
{
	int tr=layer_.transparency();
	return (100-tr)/100.;
}

void MgQLayerNode::setLayerAlpha(float alpha) 
{
	float oriAlpha=layerAlpha();
	layer_.transparency((1.-alpha)*100.);
//	if(fabs(alpha-oriAlpha) > 0.05)
//		update();
}

void MgQLayerNode::setStackLevel(int i)
{
	float z=1.1+0.01*static_cast<float>(i);
	layer_.zindex(i);
	setZValue(z); 
}

int MgQLayerNode::stackLevel()
{
	return layer_.zindex();

	//int i=static_cast<int>((zValue()-1.1) *100.);
	//return i;
}
void MgQLayerNode::saveLayerState(MgQLayerState* st)
{
	st->name_=QString::fromStdString(layer_.name());
	st->id_=QString::fromStdString(layer_.id());
	st->alpha_=layerAlpha();
	st->visible_=layerVisibility();
	st->stackLevel_=stackLevel();
}

void MgQLayerNode::updateLayer(const MgQLayerNode *node)
{
	if(node->name() != name_)
		return;

	setLayerVisibility(node->layerVisibility());
	setLayerAlpha(node->layerAlpha());
}

void MgQLayerNode::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                QWidget *widget)
{
	//Non-trasparent 
	if(layerAlpha() >= 0.99)
	{
		//Normal mode
		if(rootNodeIsChild_ == false)
		{
			addContentsBackToScene();
		}
	}
	//Transparent
	else
	{
		//Normal mode
		if(rootNodeIsChild_)
		{
			addContentsToAlphaScene();
		}

		bool clipping=(painter->clipRegion().isEmpty())?false:true;
		QRect clipRect=painter->clipRegion().boundingRect();

		float sx=painter->transform().m11();
		float sy=painter->transform().m22();

		QRectF sourceRect,targetRect;
		QRect targetRectInt;

		alphaDevice_->fill(qRgba(0,0,0,0));

		//magnifier
		if(clipping)
		{
			sourceRect=clipRect;
			targetRect=QRectF(0.,0.,clipRect.width()*sx,clipRect.height()*sy);		
		}
		else
		{
			sourceRect=layout_->boundingRect();
			targetRect=QRectF(0.,0.,sourceRect.width()*sx,sourceRect.height()*sy);
		}

     
		qDebug() <<  "LAYER-MAGNIFY" << sx << sy << layout_->boundingRect()  <<  clipRect << targetRect;

		alphaScene_->renderContents(alphaPainter_,option,targetRect,sourceRect);

		targetRectInt.setRect(targetRect.x(),targetRect.y(),targetRect.width(),targetRect.height());

		painter->setOpacity(layerAlpha());
		painter->drawImage(sourceRect,
			 alphaDevice_->copy(targetRectInt));

	}
}		

void MgQLayerNode::addContentsToAlphaScene()
{
	scene()->removeItem(rootNode_);
	rootNode_->setTransform(layout_->sceneTransform());

	if(!alphaScene_)
	{
		alphaScene_=new MgQScene;
		alphaDevice_=new QImage(823,630,QImage::Format_ARGB32);
		alphaPainter_=new QPainter(alphaDevice_);
 	}

	alphaScene_->setRootNode(rootNode_);
	rootNodeIsChild_=false;
}

void MgQLayerNode::addContentsBackToScene()
{
	if(!alphaScene_)
		return;

	alphaScene_->removeItem(rootNode_);
	rootNode_->setTransform(QTransform());
	rootNode_->setParentItem(this);
	rootNodeIsChild_=true;
}

void MgQLayerNode::addContentsToPreviewScene()
{
	scene()->removeItem(rootNode_);
	rootNode_->setTransform(layout_->sceneTransform());

	if(!previewScene_)
	{
		previewScene_=new MgQScene;
		previewDevice_=new QImage(200,200,QImage::Format_ARGB32);
		previewPainter_=new QPainter(previewDevice_);
 	}

	previewScene_->setRootNode(rootNode_);
	rootNodeIsChild_=false;
}

void MgQLayerNode::addContentsBackToSceneFromPreview()
{
	if(!previewScene_)
		return;

	previewScene_->removeItem(rootNode_);
	rootNode_->setTransform(QTransform());
	rootNode_->setParentItem(this);
	rootNodeIsChild_=true;
}

void MgQLayerNode::setStepVisible(int step,bool visible)
{
	if(steps_.count() ==0 || 
	  step < 0 || step >= steps_.count())
		return;

	if(steps_[step])
	{		
		steps_[step]->setData(MgQ::NodeIsVisibleKey,visible);
		//steps_[step]->setVisible(visible);	
	}
}

void  MgQLayerNode::clearStep(int step)
{		
	if(steps_.count() ==0 || 
	   step < 0 || step >= steps_.count())
		return;

	if(steps_[step])
	{
		delete steps_[step];
		steps_[step]=0;
	}
}

bool MgQLayerNode::stepCached(int step)
{		
	if(steps_.count() ==0 || 
	   step < 0 || step >= steps_.count())
		return false;

	if(steps_[step])
		return steps_[step]->cached();
	else
		return false;
}

MgQLayoutNode* MgQLayerNode::topLayoutNode()
{
	foreach(MgQStepNode* step,steps_)
	{
		if(step != 0)
		{
			foreach(QGraphicsItem *item, step->childItems())
			{
				if(item->data(MgQ::NodeTypeKey).isNull() == false &&
				   item->data(MgQ::NodeTypeKey).toInt() == MgQ::LayoutNode)
				{			
					MgQLayoutNode *layout=static_cast<MgQLayoutNode*>(item);
					return layout;
				}
			}
		}
	}
	return 0;		
}	

void MgQLayerNode::renderPreview()
{
	//Non-trasparent 
	if(layerAlpha() >= 0.99)
	{
		//Normal mode
		if(rootNodeIsChild_)
		{
			addContentsToPreviewScene();
		}
	}
	
	//
	//QRectF layoutRect=layout_->boundingRect();
	//QRectF targetRect((layout_->mapToScene( layoutRect).boundingRect()));
	//QRectF painterRect=painter->transform().mapRect(layout_->boundingRect());
				
	QRectF sourceRect=QRectF((layout_->mapToScene(layout_->boundingRect()).boundingRect()));

	float w=sourceRect.width();
	float h=sourceRect.height();
	float r=w/h;

	float preW=previewDevice_->width();
	float preH=previewDevice_->height();

	QRectF targetRect;
	if(h*preW/w > preH)
	{
		targetRect=QRectF(0.,0.,w*preH/h,preH);
	}
	else
	{
		targetRect=QRectF(0.,0.,preW,h*preW/w);
	}
		
	previewDevice_->fill(qRgba(255,255,255,255));
	previewScene_->renderContents(previewPainter_,targetRect,sourceRect);

	QRect targetRectInt(targetRect.x(),targetRect.y(),targetRect.width(),targetRect.height());
	if(h*100/w > 50)
	{
		previewImg_=previewDevice_->copy(targetRectInt).scaledToHeight(50,Qt::SmoothTransformation).mirrored(false,true);
	}
	else
	{
		previewImg_=previewDevice_->copy(targetRectInt).scaledToWidth(100,Qt::SmoothTransformation).mirrored(false,true);
	}


	//previewImg_.save("/var/tmp/cgr/layer.png");


	//QRect targetRectInt(targetRect.x(),targetRect.y(),targetRect.width(),targetRect.height());

	/*painter->setOpacity(layerAlpha());
	painter->drawImage(layout_->boundingRect(),
			   previewDevice_->copy(targetRectInt));*/


	addContentsBackToSceneFromPreview();

}	
	



MgQHistoNode* MgQLayerNode::histoNode(int step)
{
	if(!histoScene_)
	{
		histoScene_=new QGraphicsScene;
		histoDevice_=new QImage(200,200,QImage::Format_ARGB32);
		histoPainter_=new QPainter(histoDevice_);
 	}

	if(stepNum() == 0)
	{
		if(!histoNodes_[0])
		{
			histoNodes_[0]=new MgQHistoNode("histo");
			histoNodes_[0]->setVisible(false);
			histoScene_->addItem(histoNodes_[0]);			
		}
		return histoNodes_[0];
	}
	else
	{
		if(step < 0 || step >= steps_.count() || !steps_[step] )
			return 0;
			
		if(!histoNodes_[step])
		{
			histoNodes_[step]=new MgQHistoNode("histo");	
			histoNodes_[step]->setVisible(false);	
			histoScene_->addItem(histoNodes_[step]);
		}
		return histoNodes_[step];
	}
}


void MgQLayerNode::addHistoNode(MgQHistoNode *node)
{		
	/*if(steps_.count() ==0 || step < 0 || 
           step >= steps_.count() || !steps_[step] )
		return;*/

	/*node->scene()->removeItem(node);
	
	if(!histoScene_)
	{
		histoScene_=new QGraphicsScene;
		histoDevice_=new QImage(200,200,QImage::Format_ARGB32);
		histoPainter_=new QPainter(histoDevice_);
 	}
	
	histoScene_->addItem(node);*/
}

QPixmap MgQLayerNode::histoPixmap(int step,QSize size)
{
	MgQHistoNode *node=histoNode(step);

	if(!node || !node->cached())
		return QPixmap();

	/*if(node->pixmap().isNull() && node->cached())
	{
		return QPixmap();
	}*/

	if(node->requestedPixmapSize() == size)
	{
		return node->pixmap();
	}
	else
	{
		/*foreach(QGraphicsItem *item,histoScene_->items())
		{
			qDebug() << item << item->childItems().count() <<  item->boundingRect() << item->sceneBoundingRect();
		}

		qDebug() << histoScene_->sceneRect() << node->boundingRect() << node->sceneBoundingRect() ;*/

		if(histoDevice_->size().width() < size.width() ||
		   histoDevice_->size().height() < size.height())
		{
			
			delete histoDevice_;
			histoDevice_ = new QImage(size,QImage::Format_ARGB32);
		}

		node->setVisible(true);			
		histoDevice_->fill(qRgba(255,255,255,255));
		histoScene_->render(histoPainter_,QRectF(0,0,size.width(),size.height()));
		node->setVisible(false);

		float ratio=1;
		QRectF scr=histoScene_->sceneRect();
		if(scr.height() > 0)
			ratio=scr.width()/scr.height();
	
		QImage img=histoDevice_->copy(0,0,size.width(),size.width()/ratio+1).mirrored(false,true);

		node->setPixmap(QPixmap::fromImage(img),size);

		return node->pixmap();		
	}

	return QPixmap(); 
}
