// Formats a style rule property
function webdeveloper_formatStyleRuleProperty(styleRuleProperty)
{
    // Switch on the style rule property
    switch(styleRuleProperty)
    {
        case "margin-bottom-value":
            return "margin-bottom";
        case "margin-left-value":
            return "margin-left";
        case "margin-right-value":
            return "margin-right";
        case "margin-top-value":
            return "margin-top";
        case "padding-bottom-value":
            return "padding-bottom";
        case "padding-left-value":
            return "padding-left";
        case "padding-right-value":
            return "padding-right";
        case "padding-top-value":
            return "padding-top";
        case "-x-background-x-position":
            return "background-x-position";
        case "-x-background-y-position":
            return "background-y-position";
        default:
            return webdeveloper_sanitizeCSS(styleRuleProperty);
    }
}

// Formats a style rule value
function webdeveloper_formatStyleRuleValue(styleRuleValue)
{
    // If the style rule value is set
    if(styleRuleValue)
    {
        var rgbRegularExpression = new RegExp("rgb\\((\\d{1,3}),\\s(\\d{1,3}),\\s(\\d{1,3})\\)", "gi");
        var styleRuleValueColor  = rgbRegularExpression.exec(styleRuleValue);

        // If the style rule value is a color
        if(styleRuleValueColor)
        {
            var blue  = parseInt(styleRuleValueColor[3]).toString(16);
            var green = parseInt(styleRuleValueColor[2]).toString(16);
            var red   = parseInt(styleRuleValueColor[1]).toString(16);

            // If the blue color is only 1 character long
            if(blue.length == 1)
            {
                blue = "0" + blue;
            }

            // If the green color is only 1 character long
            if(green.length == 1)
            {
                green = "0" + green;
            }

            // If the red color is only 1 character long
            if(red.length == 1)
            {
                red = "0" + red;
            }

            return "#" + red + green + blue;
        }
    }

    return webdeveloper_sanitizeCSS(styleRuleValue);
}

// Returns the list of the style sheets for the specified document
function webdeveloper_getStyleSheetsForDocument(contentDocument, includeInline, includeAlternate)
{
    var styleSheets = new Array();

    // If the content document is set
    if(contentDocument)
    {
        var documentStyleSheets       = contentDocument.styleSheets;
        var documentStyleSheetsLength = documentStyleSheets.length;
        var documentURL               = contentDocument.documentURI;
        var styleSheet                = null;

        // Loop through the style sheets
        for(var i = 0; i < documentStyleSheetsLength; i++)
        {
            styleSheet = documentStyleSheets[i];

            // If this is a valid style sheet and including alternate style sheets or this is not alternate
            if(webdeveloper_isValidStyleSheet(styleSheet) && (includeAlternate || !webdeveloper_isAlternateStyleSheet(styleSheet)))
            {
                // If including inline style sheets or this is not inline
                if(includeInline || (styleSheet.href && styleSheet.href != documentURL))
                {
                    styleSheets.push(styleSheet);
                }

                styleSheets = styleSheets.concat(webdeveloper_getStyleSheetsImportedFromStyleSheet(styleSheet));
            }
        }
    }

    return styleSheets;
}

// Returns the list of style sheets imported from the specified style sheet
function webdeveloper_getStyleSheetsImportedFromStyleSheet(styleSheet)
{
    var styleSheets = new Array();

    // If the style sheet is set
    if(styleSheet)
    {
        var cssRule            = null;
        var cssRules           = styleSheet.cssRules;
        var cssRulesLength     = cssRules.length;
        var importedStyleSheet = null;
        var importRule         = Components.interfaces.nsIDOMCSSRule.IMPORT_RULE;

        // Loop through the the style sheet rules
        for(var i = 0; i < cssRulesLength; i++)
        {
            cssRule = cssRules[i];

            // If this is an import rule
            if(cssRule.type == importRule)
            {
                importedStyleSheet = cssRule.styleSheet;

                // If this style sheet is valid
                if(webdeveloper_isValidStyleSheet(importedStyleSheet))
                {
                    styleSheets.push(importedStyleSheet);

                    styleSheets = styleSheets.concat(webdeveloper_getStyleSheetsImportedFromStyleSheet(importedStyleSheet));
                }
            }
        }
    }

    return styleSheets;
}

// If there is a style sheet for this media type
function webdeveloper_hasStyleSheetForMedia(styleSheetList, media)
{
    // If the style sheet list and media are set
    if(styleSheetList && media)
    {
        var styleSheet       = null;
        var styleSheetLength = styleSheetList.length;

        // Loop through the style sheets
        for(var i = 0; i < styleSheetLength; i++)
        {
            styleSheet = styleSheetList[i];

            // If this style sheet is valid and is for this media type
            if(webdeveloper_isValidStyleSheet(styleSheet) && webdeveloper_isMediaStyleSheet(styleSheet, media))
            {
                return true;
            }
        }
    }

    return false;
}

// Is this style sheet an alternate style sheet
function webdeveloper_isAlternateStyleSheet(styleSheet)
{
    // If the style sheet is set
    if(styleSheet)
    {
        var ownerNode = styleSheet.ownerNode;

        // If the owner node is set
        if(ownerNode)
        {
            // If the owner node is a processing instruction
            if(ownerNode.nodeType == Components.interfaces.nsIDOMNode.PROCESSING_INSTRUCTION_NODE)
            {
                // If the processing instruction data contains alternate="yes"
                if(ownerNode.data.indexOf('alternate="yes"') != -1)
                {
                    return true;
                }
            }
            else if(ownerNode.hasAttribute("rel") && ownerNode.getAttribute("rel").toLowerCase() == "alternate stylesheet")
            {
                return true;
            }
        }
    }

    return false;
}

// Is this style sheet for this media type
function webdeveloper_isMediaStyleSheet(styleSheet, matchMediaType)
{
    // If the style sheet and match media type are set
    if(styleSheet && matchMediaType)
    {
        var media       = styleSheet.media;
        var mediaLength = media.length;
        var mediaType   = null;

        // If there is no media and the match media type is screen
        if(mediaLength == 0 && matchMediaType == "screen")
        {
            return true;
        }

        // Loop through the media
        for(var i = 0; i < mediaLength; i++)
        {
            mediaType = media.item(i).toLowerCase();

            // If the media type is all or matches the match media type
            if(mediaType == "all" || mediaType == matchMediaType)
            {
                return true;
            }
        }
    }

    return false;
}

// Is this style sheet a valid style sheet
function webdeveloper_isValidStyleSheet(styleSheet)
{
    // If the style sheet is set
    if(styleSheet)
    {
        var styleSheetHref = styleSheet.href;

        // If the style sheet href is set and this is not an internal style sheet or the style sheet href is not set and this does not have a Web Developer id
        if((styleSheetHref && styleSheetHref.indexOf("about:PreferenceStyleSheet") != 0 && styleSheetHref.indexOf("chrome://") != 0 && styleSheetHref.indexOf("data:text/css") != 0 && styleSheetHref.indexOf("jar:file://") != 0 && styleSheetHref.indexOf("resource://") != 0) || (!styleSheetHref && styleSheet.ownerNode && (!styleSheet.ownerNode.hasAttribute("id") || styleSheet.ownerNode.getAttribute("id").indexOf("webdeveloper-") != 0)))
        {
            return true;
        }
    }

    return false;
}

// Is this style rule is a valid style rule
function webdeveloper_isValidStyleRule(styleRuleList, styleRule)
{
    // If the style rule is set
    if(styleRule)
    {
        // If the style rule is an invalid style rule
        if(styleRule.indexOf("-moz-") == 0 || ((styleRule == "margin-left-ltr-source" || styleRule == "margin-left-rtl-source" || styleRule == "margin-right-ltr-source" || styleRule == "margin-right-rtl-source" || styleRule == "padding-left-ltr-source" || styleRule == "padding-left-rtl-source" || styleRule == "padding-right-ltr-source" || styleRule == "padding-right-rtl-source") && styleRuleList.getPropertyValue(styleRule) == "physical"))
        {
            return false;
        }

        return true;
    }

    return false;
}

// Sanitizes CSS to prevent XSS attacks
function webdeveloper_sanitizeCSS(css)
{
		var sanitizedCSS = null;

		// If the CSS is set
		if(css)
		{
			sanitizedCSS = css.replace("<", "&lt;").replace(">", "&gt;");
		}
		
		return sanitizedCSS;
}