/* 

        Copyright (C) 1995,96,97,98
        Free Software Foundation, Inc.

   This file is part of GNU cfengine - written and maintained 
   by Mark Burgess, Dept of Computing and Engineering, Oslo College,
   Dept. of Theoretical physics, University of Oslo
 
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

*/

/*******************************************************************/
/*                                                                 */
/*  Cfengine : remote server daemon example                        */
/*                                                                 */
/*  This uses heavy weight processes, which is not very efficient  */
/*  or pthreads if you have a working implementation               */
/*                                                                 */
/*  Mark Burgess 1997,8                                            */
/*                                                                 */
/*******************************************************************/

#define INET 1


#include "cf.defs.h"
#include "cf.extern.h"
#include "../pub/getopt.h"
#include "cfd.h"

#ifdef HAVE_TCPD_H
# include <tcpd.h>
#endif

#ifdef HAVE_PTHREAD_H
# include <pthread.h>
#endif

#ifdef HAVE_SCHED_H
# include <sched.h>
#endif

#ifdef HAVE_PTHREAD_H
pthread_attr_t PTHREADDEFAULTS;
#endif

/*******************************************************************/
/* Level 0 : Main                                                  */
/*******************************************************************/

main (argc,argv)

int argc;
char **argv;

{
CheckOptsAndInit(argc,argv);
GetNameInfo();
ParseInputFiles();

if (PARSEONLY)
   {
   exit(0);
   }

CheckVariables();
SummarizeParsing();
StartServer(argc,argv);

/* Never exit here */
}

/********************************************************************/
/* Level 1                                                          */
/********************************************************************/

CheckOptsAndInit(argc,argv)

int argc;
char **argv;

{ extern char *optarg;
 int optindex = 0;
  int c, i;

umask(0);

strcpy(VINPUTFILE,CFD_INPUT);
 
ISCFENGINE = false;   /* Switch for the parser */
PARSEONLY  = false;

InitHashTable();

AddClassToHeap("any");      /* This is a reserved word / wildcard */

while ((c=getopt_long(argc,argv,"d:f:vhpFVm",CFDOPTIONS,&optindex)) != EOF)
  {
  switch ((char) c)
      {
      case 'm': MULTITHREAD = true;
                break;

      case 'f': strcpy(VINPUTFILE,optarg);
                break;

      case 'd': 

                switch (*optarg)
                   {
                   case '1': D1 = true;
                             break;
                   case '2': D2 = true;
                             break;
                   default:  DEBUG = true;
                             break;
                   }
		
		NO_FORK = true;
		printf("cfd: Debug mode: running in foreground\n");
                break;

      case 'v': VERBOSE = true;
	        break;

      case 'V': printf("GNU %s daemon\n%s\n",CFVERSION,COPYRIGHT);
	        printf("This program is covered by the GNU Public License and may be\n");
		printf("copied free of charge. No warrenty is implied.\n\n");
                exit(0);
	        break;

      case 'p': PARSEONLY = true;
	        break;

      case 'F': NO_FORK = true;
	        break;

      default:  Syntax();
                exit(1);

      }
   }


LOGGING = true;                    /* Do output to syslog */
 
sprintf(VPREFIX,"cfd:%s:",VUQNAME); 
sprintf(VBUFF,"%s/test",LOCKFILEDIR);
MakeDirectoriesFor(VBUFF);
strcpy(VLOCKDIR,LOCKFILEDIR);

sprintf(VBUFF,"%s/test",LOGFILEDIR);

MakeDirectoriesFor(VBUFF);
strcpy(VLOGDIR,LOGFILEDIR);

strcpy(VDOMAIN,"undefined.domain");

strcpy(VCANONICALFILE,CanonifyName(VINPUTFILE));
}

/*******************************************************************/

CheckVariables()

{ struct stat statbuf;
  int i, value = -1;

#if defined HAVE_LIBPTHREAD && defined NOTBROKEN

if (MULTITHREAD)
   {
   CfLog(cfinform,"Multithreaded version","");
   }

#else
 
 CfLog(cfinform,"Single threaded version","");

#endif
  
if ((CFDSTARTTIME = time((time_t *)NULL)) == -1)
   {
   printf("Couldn't read system clock\n");
   }

 bzero(CHECKSUMDB,bufsize);

if (GetMacroValue("ChecksumDatabase"))
   {
   ExpandVarstring("$(ChecksumDatabase)",CHECKSUMDB,NULL);

   if (*CHECKSUMDB != '/')
      {
      FatalError("$(ChecksumDatabase) does not expand to an absolute filename\n");
      }
   }

if (GetMacroValue("IfElapsed"))
   {
   ExpandVarstring("$(IfElapsed)",VBUFF,NULL);
   sscanf(VBUFF,"%d",&value);

   if (value < 0)
      {
      printf("cfd: silly IfElapsed value in control\n");
      exit(1);
      }
   else
      {
      VIFELAPSED = value;
      Verbose("cfd: IfElapsed time: %d minutes\n",VIFELAPSED);
      }
   }
  
bzero(VBUFF,bufsize);

if (GetMacroValue("cfrunCommand"))
   {
   ExpandVarstring("$(cfrunCommand)",VBUFF,NULL);

   if (*VBUFF != '/')
      {
      FatalError("$(cfrunCommand) does not expand to an absolute filename\n");
      }

   sscanf(VBUFF,"%s",CFRUNCOMMAND);
   Debug("cfrunCommand is %s\n",CFRUNCOMMAND);

   if (stat(CFRUNCOMMAND,&statbuf) == -1)
      {
      FatalError("$(cfrunCommand) points to a non-existent file\n");
      }
   }

strcpy(VFQNAME,VSYSNAME.nodename);

if (GetMacroValue("MaxConnections"))
   {
   bzero(VBUFF,bufsize);
   ExpandVarstring("$(MaxConnections)",VBUFF,NULL);
   Debug("$(MaxConnections) Expanded to %s\n",VBUFF);


   CFD_MAXPROCESSES = atoi(VBUFF);

   if ((CFD_MAXPROCESSES < 1) || (CFD_MAXPROCESSES > 32))
      {
      FatalError("cfd: MaxConnections with silly value");
      }
    }
else
   {
   CFD_MAXPROCESSES = 10;
   }

if (GetMacroValue("AutoExecInterval"))
   {
   bzero(VBUFF,bufsize);
   ExpandVarstring("$(AutoExecInterval)",VBUFF,NULL);
   Debug("$(AutoExecInterval) Expanded to %s\n",VBUFF);

   CFD_INTERVAL = 60 * atoi(VBUFF);

   if ((CFD_INTERVAL < 0) || (CFD_INTERVAL > 60*24*7))
      {
      FatalError("cfd: AutoExecInterval with silly value in minutes (>0 and less than 1 week)");
      }
   }
else
   {
   CFD_INTERVAL = 0;
   }

Debug("MaxConnections = %d\n",CFD_MAXPROCESSES);
Debug("AutoExecInterval = %d\n",CFD_INTERVAL);

CHECKSUMUPDATES = true;
 
if (GetMacroValue("ChecksumUpdates") && (strcmp(GetMacroValue("ChecksumUpdates"),"off") == 0))
  {
  CHECKSUMUPDATES = false;
  } 
 
i = 0;

if (strstr(VSYSNAME.nodename,ToLowerStr(VDOMAIN)))
   {
   strcpy(VFQNAME,VSYSNAME.nodename);
   
   while(VSYSNAME.nodename[i++] != '.')
      {
      }
   
   strncpy(VUQNAME,VSYSNAME.nodename,i-1);
   }
else
   {
   sprintf(VFQNAME,"%s.%s",VSYSNAME.nodename,ToLowerStr(VDOMAIN));
   strcpy(VUQNAME,VSYSNAME.nodename);
   }
}

/*******************************************************************/

SummarizeParsing()

{ struct Auth *ptr;
  struct Item *ip;
  
if (DEBUG || D2 || D3)
   {
   printf("ACCESS GRANTED ----------------------:\n\n");

   for (ptr = VADMIT; ptr != NULL; ptr=ptr->next)
      {
      printf("Path: %s\n",ptr->path);

      for (ip = ptr->accesslist; ip != NULL; ip=ip->next)
	 {
	 printf("   Admit: %s\n",ip->name);
	 }
      }

   printf("ACCESS DENIAL ------------------------ :\n\n");

   for (ptr = VDENY; ptr != NULL; ptr=ptr->next)
      {
      printf("Path: %s\n",ptr->path);

      for (ip = ptr->accesslist; ip != NULL; ip=ip->next)
	 {
	 printf("   Deny: %s\n",ip->name);
	 }      
      }
   
   printf("Hosts denied access to :\n\n");
   }


if (ERRORCOUNT > 0)
   {
   FatalError("Execution terminated after parsing due to errors in program");
   }
}

/*******************************************************************/

StartServer(argc,argv)

int argc;
char **argv;

{ struct sockaddr_in cin, sin; 
  char recvbuffer[bufsize], sendbuffer[bufsize];
  struct servent *server;
  int sd, sd_reply, addrlen = sizeof(cin);
  struct cfd_connection *conn;
  int portnumber, yes=1, shmid;
  struct cfd_connection *NewConn();
  struct linger cflinger;
  void ExitCleanly(), AutoExec();

/* Initialize cflinger */

cflinger.l_onoff = 1;
cflinger.l_linger = 60;
  
openlog(VPREFIX,LOG_PID|LOG_NOWAIT|LOG_ODELAY,LOG_DAEMON);
  
if ((!NO_FORK) && (fork() != 0))
   {
   sprintf(OUTPUT,"cfd: starting %s\n",ctime(&CFDSTARTTIME));
   CfLog(cfinform,OUTPUT,"");
   exit(0);
   }

if (!NO_FORK)
  {
#ifdef HAVE_SETSID
  setsid();  /* BSD has arguments, SV doesn't */
#endif 
  fclose (stdin);
  fclose (stdout);
  fclose (stderr);
  closelog();
  }

signal(SIGINT,ExitCleanly);
signal(SIGTERM,ExitCleanly);
signal(SIGHUP,SIG_IGN);
signal(SIGPIPE,SIG_IGN);
signal(SIGALRM,(void *)AutoExec);

sprintf(OUTPUT,"Setting alarm to %d seconds\n",CFD_INTERVAL); 
CfLog(cfinform,OUTPUT,""); 
alarm(CFD_INTERVAL);

if ((server = getservbyname(CFENGINE_SERVICE,"tcp")) == NULL)
   {
   CfLog(cferror,"Couldn't get cfengine service","getservbyname");
   exit (1);
   }

bzero(&cin,sizeof(cin));

sin.sin_port = (unsigned short)(server->s_port); /*  Service returns network byte order */
sin.sin_addr.s_addr = INADDR_ANY;
sin.sin_family = AF_INET; 

if ((sd = socket(AF_INET,SOCK_STREAM,0)) == -1)
   {
   CfLog(cferror,"Couldn't open socket","socket");
   exit (1);
   }

if (setsockopt (sd, SOL_SOCKET, SO_REUSEADDR, (char *) &yes, sizeof (int)) == -1)
   {
   CfLog(cferror,"Couldn't set socket options","sockopt");
   exit (1);
   }

 if (setsockopt (sd, SOL_SOCKET, SO_LINGER, (char *) &cflinger, sizeof (struct linger)) == -1)
   {
   CfLog(cferror,"Couldn't set socket options","sockopt");
   exit (1);
   }


if (bind(sd,(struct sockaddr *)&sin,sizeof(sin)) == -1)  /* Must have this on server */
   {
   CfLog(cferror,"Couldn't bind to socket","bind");
   exit(1);
   }

if (listen(sd,queuesize) == -1)
   {
   CfLog(cferror,"listen failed","listen");
   exit(1);
   }

Verbose("Listening for connections on port %d..\n",server->s_port);

while (true)
   {
   if ((sd_reply = accept(sd, (struct sockaddr *)&cin, &addrlen)) == -1)
      {
      continue;
      }
   else
      {
      Debug("New connection...\n");
      conn = NewConn(sd_reply);

      while (BusyWithConnection(recvbuffer,sendbuffer,conn))
         {
         }

      Debug("End of session\n");

      CheckFileChanges(argc,argv,sd);
      }
   }
}

/*********************************************************************/
/* Level 2                                                           */
/*********************************************************************/

Syntax()

{ int i;

printf("GNU cfengine daemon: server module\n%s\n%s\n",CFVERSION,COPYRIGHT);
printf("\n");
printf("Options:\n\n");

for (i=0; CFDOPTIONS[i].name != NULL; i++)
   {
   printf("--%-20s    (-%c)\n",CFDOPTIONS[i].name,(char)CFDOPTIONS[i].val);
   }

printf("\nBug reports to bug-cfengine@prep.ai.mit.edu (News: gnu.cfengine.bug)\n");
printf("General help to help-cfengine@prep.ai.mit.edu (News: gnu.cfengine.help)\n");
printf("Info & fixes at http://www.iu.hioslo.no/~mark/cfengine.html\n");
}

/*********************************************************************/

BusyWithConnection(recvbuffer,sendbuffer,conn)

char recvbuffer[], sendbuffer[];
struct cfd_connection *conn;

  /* This is the protocol section. Here we must   */
  /* check that the incoming data are sensible    */
  /* and extract the information from the message */

{ time_t tloc, trem = 0;
  char filename[bufsize],args[bufsize];
  long time_no_see = 0;
  struct cfd_thread_arg *thr_args;

if (RecvSocketStream(conn->sd_reply,recvbuffer,bufsize,0) == -1)
   {
   return false;
   }

if (strlen(recvbuffer) == 0)
   {
   Debug("cfd: received an empty transmission!\n");
   }
  
Debug("Received: [%s] on socket %d\n",recvbuffer,conn->sd_reply);

switch (GetCommand(recvbuffer))
   {
   case cfd_exec:    bzero(args,bufsize);
                     sscanf(recvbuffer,"EXEC %[^\n]",args);

		     if (! conn->id_verified)
			{
			RefuseAccess(conn->sd_reply,sendbuffer);
			DeleteConn(conn);
			return false;
			}

		     if (!AccessControl(CFRUNCOMMAND,conn->hostname))
			{
			RefuseAccess(conn->sd_reply,sendbuffer);
			DeleteConn(conn);
			return false;			
			}

     		     if (!MatchClasses(conn,recvbuffer))
			{
			Terminate(conn->sd_reply);
			DeleteConn(conn);
			return false;
			}

		     alarm(CFD_INTERVAL); /* reset alarm */

		     DoExec(conn,sendbuffer,args);
		     Terminate(conn->sd_reply);
		     DeleteConn(conn);
		     return false;
		     
   case cfd_auth:    conn->id_verified = VerifyConnection(conn,recvbuffer+strlen("AUTH "));
                     return true; /* not finished yet */
		     
   case cfd_get:     bzero(filename,bufsize);
                     sscanf(recvbuffer,"GET %s",filename);

 	             if (! conn->id_verified)
			{
			RefuseAccess(conn->sd_reply,sendbuffer);
			DeleteConn(conn);
			return false;
			}

		     if (!AccessControl(filename,conn->hostname))
			{
			RefuseAccess(conn->sd_reply,sendbuffer);
			DeleteConn(conn);
			return false;			
			}

		     bzero(sendbuffer,bufsize);
		     
		     if ((thr_args = malloc(sizeof(struct cfd_thread_arg))) == NULL)
			{
			CfLog(cferror,"Could not allocate memory while serving file","malloc");
			return false;
			}
		     
		     thr_args->connect = conn;
		     thr_args->replybuff = sendbuffer;
		     thr_args->replyfile = strdup(filename); /* private to thread */
		     
		     SpawnCfGetFile(thr_args);
		     
		     /* close in spawn */
		     return false;
                     break;
		     
   case cfd_opendir: bzero(filename,bufsize);
                     sscanf(recvbuffer,"OPENDIR %s",filename);
		     
		     if (! conn->id_verified)
			{
			RefuseAccess(conn->sd_reply,sendbuffer);
			DeleteConn(conn);
			return false;
			}

		     if (!AccessControl(filename,conn->hostname))
			{
			RefuseAccess(conn->sd_reply,sendbuffer);
			DeleteConn(conn);
			return false;			
			}		     

		     CfOpenDirectory(conn,sendbuffer,filename);
		     DeleteConn(conn);
                     return false;
		     
   case cfd_synch:
		     if (! conn->id_verified)
			{
			RefuseAccess(conn->sd_reply,sendbuffer);
			DeleteConn(conn);
			return false;
			}
		     
                     bzero(filename,bufsize);
                     sscanf(recvbuffer,"SYNCH %ld STAT %[^\n]",&time_no_see,filename);

		     trem = (time_t) time_no_see;

                     if (time_no_see == 0 || filename[0] == '\0')
			{
			break;
			}
		     
		     if ((tloc = time((time_t *)NULL)) == -1)
                        {
                        sprintf(OUTPUT,"Couldn't read system clock\n");
			CfLog(cfinform,OUTPUT,"time");
                        }

                     if (tloc - trem > CLOCK_DRIFT)
			{
			sprintf(OUTPUT,"Clocks are too far unsynchronized %ld/%ld\n",(long)tloc,(long)trem);
			CfLog(cfinform,OUTPUT,"");
			}
		     else
			{
			sprintf(OUTPUT,"Clocks were off by %ld\n",(long)tloc-(long)trem);
			CfLog(cfverbose,OUTPUT,"");

			StatFile(conn->sd_reply,sendbuffer,filename);
			}

		     DeleteConn(conn);
		     return false;

   case cfd_md5:
		     if (! conn->id_verified)
			{
			RefuseAccess(conn->sd_reply,sendbuffer);
			return false;
			}
		     
                     bzero(filename,bufsize);
		     bzero(args,bufsize);
		     
                     CompareLocalChecksum(conn,sendbuffer,recvbuffer);
		     DeleteConn(conn);
		     return false;		     
   }

sprintf (sendbuffer,"BAD: Malformed protocol request\n%c",EOF);
CfLog(cfinform,"Malformed protocol request\n",""); 

if (send(conn->sd_reply,sendbuffer,strlen(sendbuffer)+1,0) == -1)
   {
   CfLog(cferror,"Couldn't send","send");
   exit(1);
   }

Verbose("Bad command\n");
DeleteConn(conn);
return false;
}

/**************************************************************/

CheckFileChanges(argc,argv,sd)

int argc;
char **argv;
int sd;

{ struct stat newstat;
  char filename[bufsize],line[bufsize], *sp;
  char arg[maxshellargs][bufsize];
  FILE *pp;
  int i;
  
bzero(&newstat,sizeof(struct stat));
bzero(filename,bufsize);

if ((sp=getenv(CFINPUTSVAR)) != NULL)
   {
   if (! IsAbsoluteFileName(VINPUTFILE))     /* Don't prepend to absolute names */
      { 
      strcpy(filename,sp);
      AddSlash(filename);
      }
   }

strcat(filename,VINPUTFILE);

if (stat(filename,&newstat) == -1)
   {
   sprintf(OUTPUT,"Input file %s missing!\n",filename);
   CfLog(cferror,OUTPUT,filename);
   return;
   }

Debug("Checking file updates on %s (%x/%x)\n",filename, newstat.st_ctime, CFDSTARTTIME);

if (CFDSTARTTIME < newstat.st_ctime)
   {
   sprintf(OUTPUT,"Rereading config files %s..\n",filename);
   CfLog(cfinform,OUTPUT,"");

   /* Free up memory */
   
   for (i = 0; i < hashtablesize; i++)
      {
      if (HASH[i] != NULL)
	 {
	 free(HASH[i]);
	 HASH[i] = NULL;
	 }
      }

   DeleteItemList(VHEAP);
   DeleteItemList(VNEGHEAP);
   DeleteAuthList(VADMIT);
   strcpy(VDOMAIN,"undefined.domain");
   
   VADMIT = VADMITTOP = NULL;
   VHEAP = VNEGHEAP = NULL;

   AddClassToHeap("any");
   GetNameInfo();
   ParseInputFiles();
   CheckVariables();
   alarm(CFD_INTERVAL);
   }
}

/**************************************************************/

void ExitCleanly()

{ int i;
 
sprintf(OUTPUT,"Exit -- (waiting for threads), pid = %d\n",getpid());
CfLog(cfinform,OUTPUT,"");
closelog();

#ifdef HAVE_PTHREAD_H
/* Do we care about this? 
for (i = 0; i < MAXTHREADS; i++)
   {
   if (THREADS[i] != NULL)
      {
      pthread_join(THREADS[i],(void **)NULL);
      }
   }
   */
#endif 
 
exit(0);
}

/**************************************************************/

void AutoExec()

{ char logbuffer[bufsize], line[bufsize], *sp;
  FILE *pp; 
  int print;
  
alarm(0);
VBUFF[0] = '\0';
 
ExpandVarstring("$(AutoExecCommand)",VBUFF,"");

if (strlen(VBUFF) == 0)
   {
   CfLog(cferror,"No $(AutoExecCommand) variable defined","");
   signal(SIGALRM,(void *)AutoExec);
   return;
   }

/* Use same lock as for cfrun */
 
 if (!GetLock("cfd","exec",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
   {
   sprintf(OUTPUT,"cfd: Couldn't get a lock -- too soon: IfElapsed %d, ExpireAfter %d\n",VIFELAPSED,VEXPIREAFTER);
   CfLog(cferror,OUTPUT,"");
   
   return;
   }
 
if (fork() == 0)
   {
   signal(SIGPIPE,SIG_IGN);
   
   ExpandVarstring("$(AutoExecCommand) --no-splay --inform",VBUFF,"");

   sprintf(OUTPUT,"Interval expired, executing command %s\n",VBUFF);
   CfLog(cfinform,OUTPUT,""); 

   bzero(logbuffer,bufsize);

   if ((pp = cfpopen(VBUFF,"r")) == NULL)
      {
      sprintf(OUTPUT,"Couldn't open pipe to command %s\n",VBUFF);
      CfLog(cferror,OUTPUT,"cfpopen");

      return;
      }
   
   while (!feof(pp))
      {
      if (ferror(pp))  /* abortable */
	 {
	 fflush(pp);
	 break;
	 }
      
      ReadLine(line,1,pp);
      
      if (ferror(pp))  /* abortable */
	 {
	 fflush(pp);
	 break;
	 }	 
      
      print = false;
      
      for (sp = line; *sp != '\0'; sp++)
	 {
	 if (! isspace(*sp))
	    {
	    print = true;
	    break;
	    }
	 }
      
      if (print)
	 {
	 sprintf(logbuffer,"%s\n",line);
	 CfLog(cfinform,logbuffer,"");
	 }
      }
   
   cfpclose(pp);
   closelog();
   exit(0);
   }

signal(SIGALRM,(void *)AutoExec);
alarm(CFD_INTERVAL); 
}

/**************************************************************/
/* Level 3                                                    */
/**************************************************************/

MatchClasses(conn,recvbuffer)

struct cfd_connection *conn;
char *recvbuffer;

{ char *sp;
  struct Item *classlist, *ip;
  int count = 0, n_read;

Debug("Match classes\n");

while (true)
   {
   count++;
   bzero (recvbuffer,bufsize);

   if (RecvSocketStream(conn->sd_reply, recvbuffer, bufsize,0) == -1)
      {
      if (errno == EINTR) 
         {
         continue;
         }
      }

   Debug("Got class buffer %s\n",recvbuffer);

   if (strncmp(recvbuffer,CFD_TERMINATOR,strlen(CFD_TERMINATOR)) == 0)
      {
      if (count == 1)
	 {
	 Debug("No classes were sent, assuming no restrictions...\n");
	 return true;
	 }
      
      break;
      }
   
   classlist = SplitStringAsItemList(recvbuffer,' ');

   for (ip = classlist; ip != NULL; ip=ip->next)
      {
      if (IsDefinedClass(ip->name))
	 {
	 Debug("Class %s matched, accepting...\n",ip->name);
	 DeleteItemList(classlist);
	 return true;
	 }
      
      if (strcmp(ip->name,CFD_TERMINATOR) == 0)
	 {
	 Debug("No classes matched, rejecting....\n");
	 ReplyNothing(conn);
	 DeleteItemList(classlist);
	 return false;
	 }
      }
   }

ReplyNothing(conn);
Debug("No classes matched, rejecting....\n");
DeleteItemList(classlist);
return false;
}

/**************************************************************/

DoExec(conn,sendbuffer,args)

struct cfd_connection *conn;
char *sendbuffer;
char *args;

{ char buffer[bufsize], line[bufsize], *sp;
  FILE *pp;
  int print,i;

bzero(buffer,bufsize);

if ((CFSTARTTIME = time((time_t *)NULL)) == -1)
   {
   printf("Couldn't read system clock\n");
   }

if (GetMacroValue("cfrunCommand") == NULL)
   {
   Verbose("cfd: exec request: no cfrunCommand defined\n");
   sprintf(sendbuffer,"Exec request: no cfrunCommand defined\n");
   
   if (send(conn->sd_reply,sendbuffer,bufsize,0) == -1)
      {
      CfLog(cferror,"Couldn't send","send");
      }
   return;
   }

for (sp = args; *sp != '\0'; sp++) /* Blank out -K -f */
   {
   if ((strncmp(sp,"-K",2) == 0) || (strncmp(sp,"-f",2) == 0))
      {
      *sp = ' ';
      *(sp+1) = ' ';
      }
   else if (strncmp(sp,"--no-lock",9) == 0)
      {
      for (i = 0; i < 9; i++)
	 {
	 *(sp+i) = ' ';
	 }
      }
   else if (strncmp(sp,"--file",7) == 0)
      {
      for (i = 0; i < 7; i++)
	 {
	 *(sp+i) = ' ';
	 }
      }
   }

if (!GetLock("cfd","exec",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
   {
   sprintf(sendbuffer,"cfd: Couldn't get a lock -- too soon: IfElapsed %d, ExpireAfter %d\n",VIFELAPSED,VEXPIREAFTER);

   if (send(conn->sd_reply,sendbuffer,bufsize,0) == -1)
      {
      CfLog(cferror,"Couldn't send","send");
      }
   
   return;
   }

ExpandVarstring("$(cfrunCommand) --no-splay --inform",buffer,"");

if (strlen(buffer)+strlen(args)+6 > bufsize)
   {
   sprintf(sendbuffer,"Command line too long with args: %s\n",buffer);

   if (send(conn->sd_reply,sendbuffer,bufsize,0) == -1)
      {
      CfLog(cferror,"Couldn't send","send");
      }

   ReleaseCurrentLock();
   return;
   }
else
   {
   if ((args != NULL) & (strlen(args) > 0))
      {
      strcat(buffer," ");
      strcat(buffer,args);

      sprintf(sendbuffer,"cfd: Executing %s\n",buffer);

      if (send(conn->sd_reply,sendbuffer,bufsize,0) == -1)
         {
         CfLog(cferror,"Coudn't send","send");
         }
      }
   }

sprintf(OUTPUT,"Executing command %s\n",buffer);
CfLog(cfinform,OUTPUT,""); 

bzero(sendbuffer,bufsize);

if ((pp = cfpopen(buffer,"r")) == NULL)
   {
   sprintf(OUTPUT,"Couldn't open pipe to command %s\n",buffer);
   CfLog(cferror,OUTPUT,"pipe");

   sprintf(sendbuffer,"Unable to run %s\n",buffer);

   if (send(conn->sd_reply,sendbuffer,strlen(sendbuffer),0) == -1)
      {
      CfLog(cferror,"Couldn't send","send");
      }
   ReleaseCurrentLock();
   return;
   }

while (!feof(pp))
   {
   if (ferror(pp))  /* abortable */
      {
      fflush(pp);
      break;
      }

   ReadLine(line,bufsize,pp);
   
   if (ferror(pp))  /* abortable */
      {
      fflush(pp);
      break;
      }	 

   print = false;
	 
   for (sp = line; *sp != '\0'; sp++)
      {
      if (! isspace(*sp))
	 {
	 print = true;
	 break;
	 }
      }
	 
   if (print)
      {
      sprintf(sendbuffer,"%s\n",line);

      if (send(conn->sd_reply,sendbuffer,strlen(sendbuffer),0) == -1)
         {
         CfLog(cferror,"Couldn't send","send");
	 fflush(pp);
	 break;
         }
      }
  }
      
cfpclose(pp);
ReleaseCurrentLock();
}


/**************************************************************/

GetCommand (str)

char *str;

{ int i;
  char op[bufsize];

sscanf(str,"%s",op);

for (i = 0; COMMANDS[i] != NULL; i++)
   {
   if (strcmp(op,COMMANDS[i])==0)
      {
      return i;
      }
   }

return -1;
}

/*********************************************************************/

VerifyConnection(conn,buf)

struct cfd_connection *conn;
char buf[bufsize];

{ struct sockaddr_in raddr;
  char ipstring[maxvarsize], fqname[bufsize], username[bufsize];
  char name1[bufsize], name2[bufsize], name3[bufsize];
  struct hostent *hp;
  int len, i, j, found;
   
Debug("Connecting host identifies itself as %s\n",buf);

bzero(ipstring,maxvarsize);
bzero(fqname,maxvarsize);
bzero(username,bufsize); 

sscanf(buf,"%s %s %s",ipstring,fqname,username);
len = sizeof(struct sockaddr_in);

if (getpeername(conn->sd_reply,(struct sockaddr *)&raddr,&len) == -1)
   {
   CfLog(cferror,"Couldn't get socket address\n","getpeername");
   return false;
   }

strcpy(name1,ToLowerStr(fqname));
strcpy(name2,ToLowerStr(inet_ntoa(raddr.sin_addr))); 
strcpy(name3,ToLowerStr(ipstring));
 
sprintf(OUTPUT,"Socket originates from %s=%s\n",name2,name1);
CfLog(cfverbose,OUTPUT,""); 

#ifdef HAVE_TCPD_H

if (!hosts_ctl("cfd",name1,name2,username))
   {
   Verbose("TCPwrappers rejects this packet.");
   return false;
   }

#endif

/* Do our own checking ... */


Debug("Attempting to look up hostname %s\n",name1);
 
if ((hp = gethostbyname(name1)) == NULL)
   {
   Verbose("cfd: Couldn't look up name %s\n",fqname);
   Verbose("     Make sure that fully qualified names can be looked up at your site!\n");
   Verbose("     i.e. prep.ai.mit.edu, not just prep. If you use NIS or /etc/hosts\n");
   Verbose("     make sure that the full form is registered too as an alias!\n");
   return false;
   }


    /* Now let's verify that the ip which gave us this hostname
     * is really an ip for this hostname; or is someone trying to
     * break in? (THIS IS THE CRUCIAL STEP)
     * Patch from Gunnar Gunnarsson, Ericsson */

 for (i = 0; hp->h_addr_list[i]; i++)
    {
    if (memcmp(hp->h_addr_list[i],
	       (char *) &raddr.sin_addr, sizeof(raddr.sin_addr)) == 0)
       {
       break;                     /* name is good, keep it */
       }
    }
 
 /* If we did not find it, your DNS is messed up or someone is trying
  * to pull a fast one on you. :(
  */
 
 /*   Check even the aliases list. Work around for Solaris if dns goes over NIS */
 
 if ( !hp->h_addr_list[i] )
    {
    for (j = 0; hp->h_aliases[j] !=0 ; j++)
       {
       if ( strcmp(hp->h_aliases[j],name2) == 0)
	  {
	  break;                          /* name is good, keep it */
	  }
       }
    }
 
 
 if ( !hp->h_addr_list[i] && !hp->h_aliases[j] )
    {
    sprintf(OUTPUT,"Reverse hostname lookup failed, host claiming to be %s was %s\n",buf,inet_ntoa(raddr.sin_addr));
    CfLog(cfinform,OUTPUT,"");
    return false;
    }

Debug("Host ID is %s\n",name1);
strncpy(conn->hostname,name1,maxvarsize);

Debug("User ID is %s\n",username); 
strncpy(conn->username,username,maxvarsize);
 return true;   
}

/**************************************************************/

RefuseAccess(sd,sendbuffer)

int sd;
char *sendbuffer;

{
sprintf(sendbuffer,"%s",CFFAILEDSTR);
CfLog(cfinform,"Host authentication failed\n","");

if (send(sd,sendbuffer,bufsize,0) == -1)
   {
   CfLog(cferror,"Couldn't send","send");
   }
}

/**************************************************************/

AccessControl(filename,hostname)

char *filename, *hostname;

{ struct Auth *ap;
  int access = false;
  char realname[bufsize];

bzero(realname,bufsize);

#ifdef HAVE_REALPATH
if (realpath(filename,realname) == NULL)
   {
   sprintf(OUTPUT,"Couldn't resolve filename %s from host %s\n",filename,hostname);
   CfLog(cferror,OUTPUT,"");
   return false;
   }
#else
CompressPath(realname,filename); /* in links.c */
#endif
  
Debug("AccessControl(%s,%s)\n",realname,hostname);
  
if (VADMIT == NULL)
   {
   Verbose("cfd: access list is empty, no files are visible\n");
   return false;
   }

for (ap = VADMIT; ap != NULL; ap=ap->next)
   {
   if (strncmp(ap->path,realname,strlen(ap->path)) == 0)
      {
      access = IsWildItemIn(ap->accesslist,hostname);
      break;
      }
   }

for (ap = VDENY; ap != NULL; ap=ap->next)
   {
   if (strncmp(ap->path,realname,strlen(ap->path)) == 0)
      {
      if (IsWildItemIn(ap->accesslist,hostname))
         {
         access = false;
         break;
         }
      }
   }

if (access)
   {
   sprintf(OUTPUT,"Host %s granted access to %s\n",hostname,realname);
   CfLog(cfverbose,OUTPUT,"");
   }
else
   {
   sprintf(OUTPUT,"Host %s denied access to %s\n",hostname,realname);
   CfLog(cfinform,OUTPUT,"");
   }

return access;
}

/**************************************************************/

SpawnCfGetFile(args)

struct cfd_thread_arg *args;

{
#ifdef HAVE_PTHREAD_H
 pthread_t tid;
#endif
  void *CfGetFile();
  
#if defined HAVE_LIBPTHREAD && defined NOTBROKEN

if (ACTIVE_THREADS >= CFD_MAXPROCESSES)
   {
   CfLog(cfinform,"Too many threads","");
   CfGetFile(args);
   return;
   }

if (MULTITHREAD)
   {
   Debug("Spawning new thread...\n");
   
   pthread_attr_init(&PTHREADDEFAULTS);
   pthread_attr_setdetachstate(&PTHREADDEFAULTS,PTHREAD_CREATE_DETACHED);
   
   if (MULTITHREAD && (pthread_create(&tid,&PTHREADDEFAULTS,CfGetFile,args) != 0))
      {
      CfLog(cferror,"pthread_create failed","create");
      CfGetFile(args);
      }
   

   pthread_attr_destroy(&PTHREADDEFAULTS);
   }
else
   {
   CfGetFile(args);
   return;
   }

Debug("Thread done.\n"); 
 
#else

/* Can't fork here without getting a zombie on some systems ? */
 
CfGetFile(args);

#endif

}

/**************************************************************/

StatFile(sd,sendbuffer,filename)

int sd;
char *sendbuffer, *filename;

/* Because we do not know the size or structure of remote datatypes,*/
/* the simplest way to transfer the data is to convert them into */
/* plain text and interpret them on the other side. */

{ struct cfstat cfst;
  struct stat statbuf;
  char linkbuf[bufsize];

  Debug("StatFile(%s)\n",filename);

bzero(&cfst,sizeof(struct cfstat));
  
if (strlen(filename) > maxlinksize)
   {
   sprintf(sendbuffer,"BAD: Filename suspiciously long\n");
   CfLog(cferror,sendbuffer,"");

   if (send(sd,sendbuffer,bufsize,0) == -1)
      {
      CfLog(cferror,"Couldn't send","send");
      }

   return -1;
   }

if (lstat(filename,&statbuf) == -1)
   {
   sprintf(sendbuffer,"BAD: unable to stat file %s",filename);
   CfLog(cfverbose,sendbuffer,"lstat");

   if (send(sd,sendbuffer,bufsize,0) == -1)
      {
      CfLog(cferror,"Couldn't send","send");
      }

   return -1;
   }

cfst.cf_readlink = NULL;
cfst.cf_lmode = 0;
cfst.cf_nlink = cfnosize;

bzero(linkbuf,bufsize);

if (S_ISLNK(statbuf.st_mode))
   {
   cfst.cf_type = cf_link;                   /* pointless - overwritten */
   cfst.cf_lmode = statbuf.st_mode & 07777;
   cfst.cf_nlink = statbuf.st_nlink;
       
   if (readlink(filename,linkbuf,bufsize) == -1)
      {
      sprintf(sendbuffer,"BAD: unable to read link\n");
      CfLog(cferror,sendbuffer,"readlink");
      
      if (send(sd,sendbuffer,bufsize,0) == -1)
	 {
	 CfLog(cferror,"Couldn't send","send");
	 }
      
      return -1;
      }

   Debug("readlink: %s\n",linkbuf);

   cfst.cf_readlink = linkbuf;
   }

if (stat(filename,&statbuf) == -1)
   {
   sprintf(sendbuffer,"BAD: unable to stat file %s\n",filename);
   CfLog(cfverbose,OUTPUT,"stat");

   if (send(sd,sendbuffer,bufsize,0) == -1)
      {
      CfLog(cferror,"sendbuffer","send");
      }
         
   return -1;
   }

if (S_ISDIR(statbuf.st_mode))
   {
   cfst.cf_type = cf_dir;
   }

if (S_ISREG(statbuf.st_mode))
   {
   cfst.cf_type = cf_reg;
   }

if (S_ISSOCK(statbuf.st_mode))
   {
   cfst.cf_type = cf_sock;
   }

if (S_ISCHR(statbuf.st_mode))
   {
   cfst.cf_type = cf_char;
   }

if (S_ISBLK(statbuf.st_mode))
   {
   cfst.cf_type = cf_block;
   }

if (S_ISFIFO(statbuf.st_mode))
   {
   cfst.cf_type = cf_fifo;
   }

cfst.cf_mode     = statbuf.st_mode  & 07777;      /* Make sure everything is 32 bit */
cfst.cf_uid      = statbuf.st_uid   & 0xFFFFFFFF;
cfst.cf_gid      = statbuf.st_gid   & 0xFFFFFFFF;
cfst.cf_size     = statbuf.st_size  & 0xFFFFFFFF;
cfst.cf_atime    = statbuf.st_atime & 0xFFFFFFFF;
cfst.cf_mtime    = statbuf.st_mtime & 0xFFFFFFFF;
cfst.cf_ctime    = statbuf.st_ctime & 0xFFFFFFFF;
cfst.cf_ino      = statbuf.st_ino;
cfst.cf_readlink = linkbuf;

if (cfst.cf_nlink == cfnosize)
   {
   cfst.cf_nlink = statbuf.st_nlink;
   }

#ifndef IRIX
if (statbuf.st_size > statbuf.st_blocks * DEV_BSIZE)
#else
# ifdef HAVE_ST_BLOCKS
if (statbuf.st_size > statbuf.st_blocks * DEV_BSIZE)
# else
if (statbuf.st_size > ST_NBLOCKS(statbuf) * DEV_BSIZE)
# endif
#endif
   {
   cfst.cf_makeholes = 1;   /* must have a hole to get checksum right */
   }
else
   {
   cfst.cf_makeholes = 0;
   }


bzero(sendbuffer,bufsize);

 /* send as plain text */

Debug("OK: type=%d\n mode=%o\n lmode=%o\n uid=%d\n gid=%d\n size=%ld\n atime=%d\n mtime=%d\n",
	cfst.cf_type,cfst.cf_mode,cfst.cf_lmode,cfst.cf_uid,cfst.cf_gid,(long)cfst.cf_size,
	cfst.cf_atime,cfst.cf_mtime);


sprintf(sendbuffer,"OK: %d %d %d %d %d %ld %d %d %d %d %d %d",
	cfst.cf_type,cfst.cf_mode,cfst.cf_lmode,cfst.cf_uid,cfst.cf_gid,(long)cfst.cf_size,
	cfst.cf_atime,cfst.cf_mtime,cfst.cf_ctime,cfst.cf_makeholes,cfst.cf_ino,cfst.cf_nlink);

if (send(sd,sendbuffer,bufsize,0) == -1)
   {
   CfLog(cferror,"Couldn't send","send");
   exit(1);
   }


bzero(sendbuffer,bufsize);

if (cfst.cf_readlink != NULL)
   {
   strcpy(sendbuffer,"OK:");
   strcat(sendbuffer,cfst.cf_readlink);
   }
else
   {
   sprintf(sendbuffer,"OK:");
   }

if (send(sd,sendbuffer,bufsize,0) == -1)
   {
   CfLog(cferror,"Couldn't send","send");
   exit(1);
   }

return 0;
}


/**************************************************************/

CompareLocalChecksum(conn,sendbuffer,recvbuffer)

struct cfd_connection *conn;
char *sendbuffer, *recvbuffer;

{ unsigned char digest[17], num[5],filename[bufsize];
  char *sp;
  int i;

sscanf(recvbuffer,"MD5 %s",filename);

for (sp = recvbuffer+5; *sp != ' '; sp++)
   {
   }

for (i = 0; i < 16; i++)
   {
   bzero(num,5);
   sscanf(sp,"%s",num);
   digest[i] = (char) atoi(num);
   sp += strlen((char *)num) + 1;
   }
 
Debug("CompareLocalChecksums(%s)\n",cfMDPrint(digest));
bzero(sendbuffer,bufsize);

if (ChecksumChanged(filename,digest,cfverbose,true))
   {
   sprintf(sendbuffer,"%s",CFD_TRUE);
   Debug("Checksums didn't match\n");
   
   if (send(conn->sd_reply,sendbuffer,bufsize,0) == -1)
      {
      CfLog(cferror,"","send");
      }
   
   return;
   }
else
   {
   sprintf(sendbuffer,"%s",CFD_FALSE);
   Debug("Checksums matched ok\n");
   
   if (send(conn->sd_reply,sendbuffer,bufsize,0) == -1)
      {
      CfLog(cferror,"","send");
      return;
      }
   }
}

/**************************************************************/

CfOpenDirectory(conn,sendbuffer,dirname)

struct cfd_connection *conn;
char *sendbuffer, *dirname;

{ DIR *dirh;
  struct dirent *dirp;
  int offset;

Debug("CfOpenDirectory(%s)\n",dirname);
  
if (*dirname != '/')
   {
   sprintf(sendbuffer,"BAD: request to access a non-absolute filename\n");
   return -1;
   }

if ((dirh = opendir(dirname)) == NULL)
   {
   Debug("cfengine, couldn't open dir %s\n",dirname);
   sprintf(sendbuffer,"BAD: cfengine, couldn't open dir %s\n",dirname);
   if (send(conn->sd_reply,sendbuffer,bufsize,0) == -1)
      {
      CfLog(cferror,"","send");
      exit(1);
      }
   return -1;
   }

/* Pack names for transmission */

bzero(sendbuffer,bufsize);

offset = 0;

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (strlen(dirp->d_name)+1+offset >= bufsize - buffer_margin)
      {
      if (send(conn->sd_reply,sendbuffer,bufsize,0) == -1)
         {
         CfLog(cferror,"","send");
         exit(1);
	 }

      offset = 0;
      bzero(sendbuffer,bufsize);
      }

   strcpy(sendbuffer+offset,dirp->d_name);
   offset += strlen(dirp->d_name) + 1;     /* + zero byte separator */

   /* could send the stat value here */
   }

if (send(conn->sd_reply,sendbuffer,bufsize,0) == -1)
   {
   CfLog(cferror,"","send");
   exit(1);
   }

closedir(dirh);
return 0;
}

/***************************************************************/

Terminate(sd)

int sd;

{ char buffer[bufsize];

bzero(buffer,bufsize);

strcpy(buffer,CFD_TERMINATOR);

if (send(sd,buffer,strlen(buffer)+1,0) == -1)
   {
   CfLog(cferror,"","send");
   Verbose("Unable to reply with terminator...\n");
   }
}

/***************************************************************/

DeleteAuthList(ap)

struct Auth *ap;

{
if (ap != NULL)
   {
   DeleteAuthList(ap->next);
   ap->next = NULL;

   DeleteItemList(ap->accesslist);

   free((char *)ap);
   }
}

/***************************************************************/
/* Level 4                                                     */
/***************************************************************/

void *CfGetFile(args)

struct cfd_thread_arg *args;

{ int sd,fd, n_read;
  uid_t uid;
  char sendbuffer[bufsize], *filename;
  struct stat statbuf;
  struct passwd *pw;
#ifdef HAVE_UTIME_H
  struct utimbuf timebuf;
#endif
  
#if defined HAVE_LIBPTHREAD && defined NOTBROKEN  
  pthread_mutex_t mutex;

if (MULTITHREAD && (pthread_mutex_lock(&mutex) != 0))
   {
   CfLog(cferror,"pthread_mutex_lock failed","pthread_mutex_lock");
   free(args->replyfile);  /* from strdup in each thread */
   DeleteConn(args->connect);
   free((char *)args);
   return NULL;
   }

ACTIVE_THREADS++;

if (MULTITHREAD && (pthread_mutex_unlock(&mutex) != 0))
   {
   CfLog(cferror,"pthread_mutex_unlock failed","unlock");
   }
  
#endif

sd         = (args->connect)->sd_reply;
filename   = args->replyfile;

if ((pw=getpwnam((args->connect)->username)) == NULL)
   {
   uid = -2;
   }
else
   {
   uid = pw->pw_uid;
   }

Debug("CfGetFile(%s on sd=%d)\n",filename,sd);

stat(filename,&statbuf);

/* Now check to see if we have remote permission */

if (statbuf.st_uid == uid)
   {
   Debug("Caller %s is the owner of the file\n",(args->connect)->username);
   }
else
   {
   /* We are not the owner of the file and we don't care about groups */
   if (statbuf.st_mode & S_IROTH)
      {
      Debug("Caller %s not owner of the file but permission granted\n",(args->connect)->username);
      }
   else
      {
      Debug("Caller %s not owner of the file permission denied\n",(args->connect)->username);
      RefuseAccess(args->connect->sd_reply,sendbuffer);
      DeleteConn(args->connect);
 
      free(args->replyfile);  /* from strdup in each thread */ 
      free((char *)args);
      return;
      }
   }

 if ((fd = open(filename,O_RDONLY)) == -1)
   {
   sprintf(sendbuffer,"BAD: can't access %s on server!\n",filename);
   CfLog(cferror,sendbuffer,"open");
   }
else
   {
   while(true)
      {
      bzero(sendbuffer,bufsize);
      
      if ((n_read = read(fd,sendbuffer,bufsize)) == -1)
	 {
	 close(fd);
	 CfLog(cferror,"read failed in GetFile","read");
	 break;
	 }
      
      if (n_read == 0)
	 {
	 break;
	 }
      
      if (send(sd,sendbuffer,n_read,0) == -1)
	 {
	 close(fd);
	 CfLog(cferror,"Send failed in GetFile","send");
	 break;
	 }
      }
   
   close(fd);
   }


DeleteConn(args->connect);
 
free(args->replyfile);  /* from strdup in each thread */ 
free((char *)args);
 
#if defined HAVE_LIBPTHREAD && defined NOTBROKEN
 
if (pthread_mutex_lock(&mutex) != 0)
   {
   CfLog(cferror,"pthread_mutex_lock failed","pthread_mutex_lock");
   return;
   }
 
ACTIVE_THREADS--;
   
if (pthread_mutex_unlock(&mutex) != 0)
   {
   CfLog(cferror,"pthread_mutex_unlock failed","unlock");
   }

#endif 

return NULL;
}

/***************************************************************/

ReplyNothing(conn)

struct cfd_connection *conn;

{ char buffer[bufsize];

sprintf(buffer,"Hello %s, nothing to do here...\n",conn->hostname,VFQNAME);

if (send(conn->sd_reply,buffer,bufsize,0) == -1)
   {
   CfLog(cferror,"","send");
   }
}



/***************************************************************/
/* Toolkit/Class: conn                                         */
/***************************************************************/

struct cfd_connection *NewConn(sd)  /* construct */

int sd;

{ struct cfd_connection *conn;

conn = (struct cfd_connection *) malloc(sizeof(struct cfd_connection));

conn->sd_reply = sd;
conn->id_verified = false;
conn->hostname[0] = '\0'; 
 
return conn;
}

/***************************************************************/

DeleteConn(conn) /* destruct */

struct cfd_connection *conn;

{
close(conn->sd_reply);
free ((char *)conn);
}

/* EOF */



