/* 

        Copyright (C) 1995
        Free Software Foundation, Inc.

   This file is part of GNU cfengine - written and maintained 
   by Mark Burgess, Dept of Computing and Engineering, Oslo College,
   Dept. of Theoretical physics, University of Oslo
 
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version. 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

*/

/*******************************************************************/
/*                                                                 */
/*  Cfengine : a site configuration langugae                       */
/*                                                                 */
/*  Module: (main) cfengine.c                                      */
/*                                                                 */
/*  Mark Burgess 1994/96                                           */
/*                                                                 */
/*******************************************************************/

#define INET

#include "cf.defs.h"
#include "cf.extern.h"
#include "../pub/getopt.h"

/*******************************************************************/
/* Level 0 : Main                                                  */
/*******************************************************************/

main (argc,argv)

char *argv[];
int argc;

{ void HandleSignal();
  struct Item *action;

signal (SIGTERM,HandleSignal);                   /* Signal Handler */
signal (SIGHUP,HandleSignal);
signal (SIGINT,HandleSignal);
signal (SIGPIPE,HandleSignal);

Initialize(argc,argv);

if (! NOHARDCLASSES)
   {
   GetNameInfo();
   }

PreNetConfig();

ReadRCFile(); /* Should come before parsing so that it can be overridden */

ParseInputFiles();

EchoValues();

if (PARSEONLY)                            /* Establish lock for root */
   {
   exit(0);
   }
 
if (PRSYSADM)                                           /* -a option */
   {
   printf("%s\n",VSYSADM);
   exit (0);
   }


openlog(VPREFIX,LOG_PID|LOG_NOWAIT|LOG_ODELAY,LOG_USER);

CheckSystemVariables();
 
for (action = VACTIONSEQ; action !=NULL; action=action->next)
   {
   Debug("Looking at action %s\n\n",action->name);
   
   switch(EvaluateAction(action->name,&VADDCLASSES))
      {
      case mountinfo:
                        GetHomeInfo();
                        GetMountInfo();
                        break;
      case mkpaths:
                        if (! NOFILECHECK)
                           {
			   GetSetuidLog();
			   MakePaths();
			   SaveSetuidLog();
			   }
                        break;
      case lnks:
	  	        if (!GetLock(ASUniqueName("link"),"all",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
                           {
                           continue;
                           }
			
                        MakeChildLinks();
                        MakeLinks();
			ReleaseCurrentLock();
                        break;
      case simplelnks:

	  	        if (!GetLock(ASUniqueName("link"),"simple",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
                           {
                           continue;
                           }
	  
                        MakeLinks();
			ReleaseCurrentLock();
                        break;
      case childlnks:

	  	        if (!GetLock(ASUniqueName("link"),"child",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
                           {
                           continue;
                           }
			
                        MakeChildLinks();
			ReleaseCurrentLock();
                        break;
      case chkmail:
			if (!GetLock("Mount",CanonifyName(VFSTAB[VSYSTEMHARDCLASS]),0,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
			   {
			   return; /* Note IFELAPSED must be zero to avoid conflict with mountresc */
			   }
	  
                        MailCheck();

			ReleaseCurrentLock();
                        break;
      case mountall:
                        if (! NOMOUNTS )
                           {
                           MountFileSystems();
                           }
                        break;
      case requir:
                        CheckRequired();
                        break;
      case tidyf:
                        if (! NOTIDY) 
                           {
			   if (!GetLock(ASUniqueName("tidy"),"",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
			      {
			      continue;
			      }
			   
                           TidyFiles();
			   ReleaseCurrentLock();
                           }
                        break;
      case shellcom:
                        if (! NOSCRIPTS)
                           {
                           Scripts();
                           }
                        break;
      case chkfiles:
                        if (! NOFILECHECK)
                           {
			   if (!GetLock(ASUniqueName("files"),"",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
			      {
			      continue;
			      }
			   
                           GetSetuidLog();
                           CheckFiles();
                           SaveSetuidLog();
			   ReleaseCurrentLock();
                           }
                        break;
      case disabl:
	  	        if (!GetLock(ASUniqueName("disable"),"",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
                           {
                           continue;
                           }
			
                        DisableFiles();
			ReleaseCurrentLock();
                        break;
      case mountresc:
                        if (! NOMOUNTS)
                           {
			   if (!GetLock("Mountres",CanonifyName(VFSTAB[VSYSTEMHARDCLASS]),0,VEXPIREAFTER,VUQNAME,0))
			      {
			      return;    /* Note IFELAPSED must be zero to avoid conflict with mountresc */
			      }

			   Banner("Mount file system resources");
                           MountHomeBinServers();
                           MountMisc();
			   ReleaseCurrentLock();
                           }
                        break;
      case umnt:
	  	        if (!GetLock(ASUniqueName("unmount"),"",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
                           {
                           continue;
                           }
			
                        Unmount();
			ReleaseCurrentLock();
                        break;

      case edfil:
                        if (! NOEDITS)
                           {
                           EditFiles();
                           }
                        break;


      case resolv:

	                if (!GetLock(ASUniqueName("resolve"),"",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
                           {
                           continue;
                           }
			
                        CheckResolv();
			ReleaseCurrentLock();
                        break;
      case imag:
                        if (! NOCOPY)
                           {
			   GetSetuidLog();
                           MakeImages();
			   SaveSetuidLog();
                           }
                        break;
      case netconfig:

			if (IFCONF)
			   {
			   ConfigureInterfaces();
			   }

                        break;

      case tzone:
                        CheckTimeZone();
                        break;

      case procs:
                        if (! NOPROCS)
			   {
			   if (!GetLock(ASUniqueName("processes"),"",VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
			      {
			      continue;
			      }
			   
			   CheckProcesses();
			   ReleaseCurrentLock();
			   }
                        break;

      case plugin:      break;

      default:  
                        sprintf(VBUFF,"Undefined action %s in sequence\n",action->name);
                        FatalError(VBUFF);
                        break;
      }

   DeleteItemList(VADDCLASSES);
   VADDCLASSES = NULL;
   }

closelog();
exit(0); 
}

/*******************************************************************/
/* Level 1                                                         */
/*******************************************************************/

Initialize(argc,argv)

char *argv[];
int argc;

{ char *sp, *cfargv[maxargs];
  int i, cfargc;
  struct stat statbuf;
  
umask(0);                 /* This make the DEFAULT modes absolute */

strcpy(VDOMAIN,CF_START_DOMAIN);

strcpy(VERSION,CFVERSION);

ISCFENGINE = true;

VFACULTY[0] = '\0';
VSYSADM[0] = '\0';
VNETMASK[0]= '\0';
VBROADCAST[0] = '\0';
VMAILSERVER[0] = '\0';
VREPOSITORY[0] = '\0';
VDEFAULTROUTE[0] = '\0';
VARCH[0] = '\0';

ALLCLASSBUFFER[0] = '\0';
CURRENTITEM[0] = '\0';
GROUPBUFF[0] = '\0';
ACTIONBUFF[0] = '\0';
CURRENTPATH[0] = '\0';
CLASSBUFF[0] = '\0';
LINKFROM[0] = '\0';

re_syntax_options |= RE_INTERVALS;

strcpy(VINPUTFILE,"cfengine.conf");
strcpy(VNFSTYPE,"nfs");

InitHashTable();

AddClassToHeap("any");      /* This is a reserved word / wildcard */

if (stat("/etc/debian_version",&statbuf) != -1)
   {
   Verbose("\nThis appears to be a debian system.\n");
   AddClassToHeap("debian");
   }

if (stat("/usr/src/redhat",&statbuf) != -1)
   {
   Verbose("This appears to be a redhat system.\n");
   AddClassToHeap("redhat");
   }

/* Note we need to fix the options since the argv mechanism doesn't */
 /* work when the shell #!/bla/cfengine -v -f notation is used.      */
 /* Everything ends up inside a single argument! Here's the fix      */

cfargc = 1;
cfargv[0]="cfengine";

for (i = 1; i < argc; i++)
   {
   sp = argv[i];

   while (*sp != '\0')
      {
      while (*sp == ' ' && *sp != '\0') /* Skip to arg */
         {
	 if (*sp == ' ')
	    {
	    *sp = '\0'; /* Break argv string */
	    }
	 sp++;
         }

      cfargv[cfargc++] = sp;
      
      while (*sp != ' ' && *sp != '\0') /* Skip to white space */
         {
	 sp++;
         }
      }
   }

VDEFAULTBINSERVER.name = "";

VEXPIREAFTER = VDEFAULTEXPIREAFTER;
VIFELAPSED = VDEFAULTIFELAPSED;

CheckOpts(cfargc,cfargv);
}

/*******************************************************************/

PreNetConfig()                           /* Execute a shell script */

{ struct stat buf;
  char comm[bufsize];
  char *sp;
  FILE *pp;

if (NOPRECONFIG)
   {
   CfLog(cfverbose,"Ignoring the cf.preconf file: option set","");
   return;
   }

if ((sp=getenv(CFINPUTSVAR)) != NULL)
   {
   sprintf(comm,"%s/%s",sp,VPRECONFIG);

   if (stat(comm,&buf) == -1)
       {
       CfLog(cfverbose,"No preconfiguration file","");
       return;
       }
   
   sprintf(comm,"%s/%s %s 2>&1",sp,VPRECONFIG,CLASSTEXT[VSYSTEMHARDCLASS]);
   }
else
   {
   sprintf(comm,"./%s",VPRECONFIG);

   if (stat(comm,&buf) == -1)
       {
       CfLog(cfverbose,"No preconfiguration file\n","");
       return;
       }
      
   sprintf(comm,"./%s %s",VPRECONFIG,CLASSTEXT[VSYSTEMHARDCLASS]);
   }

if (S_ISDIR(buf.st_mode) || S_ISCHR(buf.st_mode) || S_ISBLK(buf.st_mode))
   {
   sprintf(OUTPUT,"Error: %s was not a regular file\n",VPRECONFIG);
   CfLog(cferror,OUTPUT,"");
   FatalError("Aborting.");
   }

Verbose("\n\nExecuting Net Preconfiguration script...%s\n\n",VPRECONFIG);
 
if ((pp = cfpopen(comm,"r")) == NULL)
   {
   sprintf(OUTPUT,"Failed to open pipe to %s\n",comm);
   CfLog(cferror,OUTPUT,"");
   return;
   }

while (!feof(pp))
   {
   if (ferror(pp))  /* abortable */
      {
      CfLog(cferror,"Error running preconfig\n","ferror");
      break;
      }

   ReadLine(VBUFF,bufsize,pp);

   if (feof(pp))
      {
      break;
      }
   
   if (ferror(pp))  /* abortable */
      {
      CfLog(cferror,"Error running preconfig\n","ferror");
      break;
      }

   CfLog(cfinform,VBUFF,"");
   }

cfpclose(pp);
}

/*******************************************************************/

ReadRCFile()

{ char filename[bufsize], buffer[bufsize], *sp, *mp;
  char class[maxvarsize], variable[maxvarsize], value[maxvarsize];
  int c;
  FILE *fp;

filename[0] = buffer[0] = class[0] = variable[0] = value[0] = '\0';
LINENUMBER = 0;

if ((sp=getenv(CFINPUTSVAR)) != NULL)
   {
   strcpy(filename,sp);
   if (filename[strlen(filename)-1] != '/')
      {
      strcat(filename,"/");
      }
   }

strcat(filename,VRCFILE);

if ((fp = fopen(filename,"r")) == NULL)      /* Open root file */
   {
   return;
   }

while (!feof(fp))
   {
   ReadLine(buffer,bufsize,fp);
   LINENUMBER++;
   class[0]='\0';
   variable[0]='\0';
   value[0]='\0';

   if (strlen(buffer) == 0 || buffer[0] == '#')
      {
      continue;
      }

   if (strstr(buffer,":") == 0)
      {
      sprintf(OUTPUT,"Malformed line (missing :) in resource file %s - skipping\n",VRCFILE);
      CfLog(cferror,OUTPUT,"");
      continue;
      }

   sscanf(buffer,"%[^.].%[^:]:%[^\n]",class,variable,value);

   if (class[0] == '\0' || variable[0] == '\0' || value[0] == '\0')
      {
      sprintf(OUTPUT,"%s:%s - Bad resource\n",VRCFILE,buffer);
      CfLog(cferror,OUTPUT,"");
      sprintf(OUTPUT,"class=%s,variable=%s,value=%s\n",class,variable,value);
      CfLog(cferror,OUTPUT,"");
      FatalError("Bad resource");
      }

   if (strcmp(CLASSTEXT[VSYSTEMHARDCLASS],class) != 0) 
      {
      continue;  /* No point if not equal*/
      }

   if ((mp = strdup(value)) == NULL)
      {
      perror("malloc");
      FatalError("malloc in ReadRCFile");
      }

   sprintf(OUTPUT,"Redefining resource %s as %s (%s)\n",variable,value,class);
   CfLog(cfverbose,OUTPUT,"");

   c = VSYSTEMHARDCLASS;

   switch (GetResource(variable))
      {
      case rmountcom:
                        VMOUNTCOMM[c] = mp;
                        break;

      case runmountcom:
                        VUNMOUNTCOMM[c] = mp;
                        break;
      case rethernet:
                        VIFDEV[c] = mp;
                        break;
      case rmountopts:
                        VMOUNTOPTS[c] = mp;
                        break;
      case rfstab:
                        VFSTAB[c] = mp;
                        break;
      case rmaildir:
                        VMAILDIR[c] = mp;
                        break;
      case rnetstat:
                        VNETSTAT[c] = mp;
                        break;
      case rpscomm:
	                VPSCOMM[c] = mp;
	                break;
      case rpsopts:
	                VPSOPTS[c] = mp;
	                break;
      default:
                        sprintf(VBUFF,"Bad resource %s in %s\n",variable,VRCFILE);
                        FatalError(VBUFF);
                        break;
      }
   }

fclose(fp);
}

/*******************************************************************/

EchoValues()

{ struct Item *ip;
  int n = 0;
  
if (strstr(VSYSNAME.nodename,ToLowerStr(VDOMAIN)))
   {
   strcpy(VFQNAME,VSYSNAME.nodename);
   
   while(VSYSNAME.nodename[n++] != '.')
      {
      }
   
   strncpy(VUQNAME,VSYSNAME.nodename,n-1);
   }
else
   {
   sprintf(VFQNAME,"%s.%s",VSYSNAME.nodename,ToLowerStr(VDOMAIN));
   strcpy(VUQNAME,VSYSNAME.nodename);
   }

bzero(VBUFF,bufsize);

if (GetMacroValue("OutputPrefix"))
   {
   ExpandVarstring("$(OutputPrefix)",VBUFF,NULL);
   }

if (strlen(VBUFF) != 0)
   {
   strncpy(VPREFIX,VBUFF,40);  /* No more than 40 char prefix (!) */
   }
else
   {
   strcpy(VPREFIX,"cfengine:");
   strcat(VPREFIX,VUQNAME);
   }


sprintf(LOGFILE,"%s/cfengine.%s.log",LOGFILEDIR,VUQNAME);

strcpy(VSETUIDLOG,LOGFILE);

if (VERBOSE || DEBUG || D2 || D3)
   {
   struct Item *ip;
   
   ListDefinedClasses();

   printf("\nGlobal expiry time for locks: %d minutes\n",VEXPIREAFTER);
   printf("\nGlobal anti-spam elapse time: %d minutes\n\n",VIFELAPSED);

   printf("Extensions which should not be directories = ( ");
   for (ip = EXTENSIONLIST; ip != NULL; ip=ip->next)
      {
      printf("%s ",ip->name);
      }
   printf(")\n");

   printf("Suspicious filenames to be warned about = ( ");
   for (ip = SUSPICIOUSLIST; ip != NULL; ip=ip->next)
      {
      printf("%s ",ip->name);
      }
   printf(")\n");
   
   }

if (DEBUG || D2 || D3)
   {
   printf("\nFully qualified hostname is: %s\n",VFQNAME);
   printf("Unqualified hostname is: %s\n",VUQNAME);
   printf("\nSystem administrator mail address is: %s\n",VSYSADM);
   printf("Sensible size = %d\n",SENSIBLEFSSIZE);
   printf("Sensible count = %d\n",SENSIBLEFILECOUNT);
   printf("Edit File (Max) Size = %d\n\n",EDITFILESIZE);
   printf("------------------------------------------------------------\n");
   ListDefinedInterfaces();
   printf("------------------------------------------------------------\n");
   ListDefinedBinservers();
   printf("------------------------------------------------------------\n");
   ListDefinedHomeservers();
   printf("------------------------------------------------------------\n");
   ListDefinedHomePatterns();
   printf("------------------------------------------------------------\n");
   ListActionSequence();
   printf("\nUsing mailserver %s\n",VMAILSERVER);
   printf("\nLocal mountpoints: ");
   for (ip = VMOUNTLIST; ip != NULL; ip=ip->next)
      {
      printf ("%s ",ip->name);
      }
   printf("\n");
   printf("\nDefault route for packets %s\n\n",VDEFAULTROUTE);
   printf("\nFile repository = %s\n\n",VREPOSITORY);
   printf("\nNet interface name = %s\n",VIFDEV[VSYSTEMHARDCLASS]);
   printf("------------------------------------------------------------\n");
   ListDefinedResolvers();
   printf("------------------------------------------------------------\n");
   ListDefinedRequired();
   printf("------------------------------------------------------------\n");
   ListDefinedMountables();
   printf("------------------------------------------------------------\n");
   ListMiscMounts();
   printf("------------------------------------------------------------\n");
   ListUnmounts();
   printf("------------------------------------------------------------\n");
   ListDefinedMakePaths();
   printf("------------------------------------------------------------\n");
   ListDefinedImports();
   printf("------------------------------------------------------------\n");
   ListFiles();
   printf("------------------------------------------------------------\n");
   ListACLs();
   printf("------------------------------------------------------------\n");
   ListDefinedIgnore();
   printf("------------------------------------------------------------\n");
   ListFileEdits();
   printf("------------------------------------------------------------\n");
   ListProcesses();
   printf("------------------------------------------------------------\n");
   ListDefinedImages();
   printf("------------------------------------------------------------\n");
   ListDefinedTidy();
   printf("------------------------------------------------------------\n");
   ListDefinedDisable();
   printf("------------------------------------------------------------\n");
   ListDefinedLinks();
   printf("------------------------------------------------------------\n");
   ListDefinedLinkchs();
   printf("------------------------------------------------------------\n");
   ListDefinedScripts();
   printf("------------------------------------------------------------\n");

   
   if (IGNORELOCK)
      {
      printf("\nIgnoring locks...\n");
      }
   }
}

/*******************************************************************/

CheckSystemVariables()

{ char id[maxvarsize];
  int time, hash, activecfs, locks;
  char *sp;

Debug2("\n\n");

if (VACTIONSEQ == NULL)
   {
   Warning("actionsequence is empty ");
   Warning("perhaps cfengine.conf has not yet been set up?");
   }

sprintf(id,"%d",geteuid());   /* get effective user id */

if (VACCESSLIST != NULL && !IsItemIn(VACCESSLIST,id))
   {
   FatalError("Access denied");
   }

Debug2("cfengine -d : Debugging output enabled.\n");

if (DONTDO && (VERBOSE || DEBUG || D2))
   {
   printf("cfengine -n: Running in ``All talk and no action'' mode\n");
   }

if (TRAVLINKS && (VERBOSE || DEBUG || D2))
   {
   printf("cfengine -l : will traverse symbolic links\n");
   }

if ( ! IFCONF && (VERBOSE || DEBUG || D2))
   {
   printf("cfengine -i : suppressing interface configuration\n");
   }

if ( NOFILECHECK && (VERBOSE || DEBUG || D2))
   {
   printf("cfengine -c : suppressing file checks\n");
   }

if ( NOSCRIPTS && (VERBOSE || DEBUG || D2))
   {
   printf("cfengine -s : suppressing script execution\n");
   }

if ( NOTIDY && (VERBOSE || DEBUG || D2))
   {
   printf("cfengine -t : suppressing tidy function\n");
   }

if ( NOMOUNTS && (VERBOSE || DEBUG || D2))
   {
   printf("cfengine -m : suppressing mount operations\n");
   }

if ( MOUNTCHECK && (VERBOSE || DEBUG || D2))
   {
   printf("cfengine -C : check mount points\n");
   }


if (ERRORCOUNT > 0)
   {
   FatalError("Execution terminated after parsing due to errors in program");
   }

strcpy(VCANONICALFILE,CanonifyName(VINPUTFILE));

if (getuid() != 0)
   {
   Verbose("\n(Not root user...)\n\n");
   
   if ((sp = getenv("HOME")) == NULL)
      {
      FatalError("You do not have a HOME variable pointing to your home directory");
      }

   sprintf(VLOGDIR,"%s/.cfengine",sp);
   sprintf(VLOCKDIR,"%s/.cfengine",sp);

   sprintf(VBUFF,"%s/.cfengine/test",sp);
   MakeDirectoriesFor(VBUFF);
   }
else
   {
   strcpy(VLOGDIR,LOGFILEDIR);
   strcpy(VLOCKDIR,LOCKFILEDIR);

   sprintf(VBUFF,"%s/test",LOCKFILEDIR);
   MakeDirectoriesFor(VBUFF);
   sprintf(VBUFF,"%s/test",LOGFILEDIR);
   MakeDirectoriesFor(VBUFF);
   }

if (GetMacroValue("MaxCfengines"))
   {
   activecfs = atoi(GetMacroValue("MaxCfengines"));
   locks = CountActiveLocks();
   
   if (locks >= activecfs)
      {
      sprintf(OUTPUT,"Too many cfengines running (%d/%d)\n",locks,activecfs);
      CfLog(cferror,OUTPUT,"");
      closelog();
      exit(1);
      }
   }

if (GetMacroValue("Verbose") && (strcmp(GetMacroValue("Verbose"),"on") == 0))
   {
   VERBOSE = true;
   }


if (GetMacroValue("Inform") && (strcmp(GetMacroValue("Inform"),"on") == 0))
   {
   INFORM = true;
   } 

INFORM_save = INFORM;
LOGGING_save = LOGGING;
 
if (GetMacroValue("Syslog") && (strcmp(GetMacroValue("Syslog"),"on") == 0))
   {
   LOGGING = true;
   }

if (GetMacroValue("DryRun") && (strcmp(GetMacroValue("DryRun"),"on") == 0))
   {
   DONTDO = true;
   }
 
if (GetMacroValue("Warnings") && (strcmp(GetMacroValue("Warnings"),"on") == 0))
   {
   WARNINGS = true;
   }

if (GetMacroValue("NonAlphaNumFiles") && (strcmp(GetMacroValue("NonAlphaNumFiles"),"on") == 0))
   {
   NONALPHAFILES = true;
   }

if (GetMacroValue("SecureInput") && (strcmp(GetMacroValue("SecureInput"),"on") == 0))
   {
   CFPARANOID = true;
   }

if (GetMacroValue("ShowActions") && (strcmp(GetMacroValue("ShowActions"),"on") == 0))
   {
   SHOWACTIONS = true;
   }
 
bzero(CHECKSUMDB,bufsize);

if (GetMacroValue("ChecksumDatabase"))
   {
   ExpandVarstring("$(ChecksumDatabase)",CHECKSUMDB,NULL);

   if (*CHECKSUMDB != '/')
      {
      FatalError("$(ChecksumDatabase) does not expand to an absolute filename\n");
      }
   }

if (GetMacroValue("ChecksumUpdates") && (strcmp(GetMacroValue("ChecksumUpdates"),"on") == 0))
   {
   CHECKSUMUPDATES = true;
   } 
 
if (GetMacroValue("TimeOut"))
   {
   time = atoi(GetMacroValue("TimeOut"));

   if (time < 3 || time > 60)
      {
      CfLog(cfinform,"TimeOut not between 3 and 60 seconds, ignoring.\n","");
      return;
      }

   CF_TIMEOUT = time;
   }

 if (NOSPLAY)
   {
   return;
   }

time = 0;
hash = Hash(VFQNAME);
 
if (GetMacroValue("SplayTime"))
   {
   time = atoi(GetMacroValue("SplayTime"));

   if (time < 0)
      {
      CfLog(cfinform,"SplayTime with negative value, ignoring.\n","");
      return;
      }

   sprintf(OUTPUT,"Sleeping for SplayTime %d seconds\n\n",(int)(time*60*hash/hashtablesize));
   CfLog(cfverbose,OUTPUT,"");
   sleep((int)(time*60*hash/hashtablesize));
   }

}

/*******************************************************************/

enum aseq EvaluateAction(action,classlist)

char *action;
struct Item **classlist;

{ int i,j = 0;
  char *sp,cbuff[bufsize],actiontxt[bufsize];
  struct Item *ip;

cbuff[0]='\0';
actiontxt[0]='\0';
sp = action;

while (*sp != '\0')
   {
   ++j;
   sscanf(sp,"%[^.]",cbuff);

   while ((*sp != '\0') && (*sp !='.'))
      {
      sp++;
      }
 
   if (*sp == '.')
      {
      sp++;
      }
 
   if (IsHardClass(cbuff))
      {
      sprintf(OUTPUT,"Error in action sequence: %s\n",action);
      CfLog(cferror,OUTPUT);
      FatalError("You cannot add a reserved class!");
      }
 
   if (j == 1)
      {
      VIFELAPSED = VDEFAULTIFELAPSED;
      VEXPIREAFTER = VDEFAULTEXPIREAFTER;
      strcpy(actiontxt,cbuff);
      continue;
      }
   else
      {
      if ((strncmp(actiontxt,"module:",7) != 0) && ! IsSpecialClass(cbuff))
	 {
         AppendItem(classlist,cbuff,NULL);
	 }
      }
   }


BuildClassEnvironment();

if ((VERBOSE || DEBUG || D2) && *classlist != NULL)
   {
   printf("\n                  New temporary class additions\n");
   printf("                  -----------------------------\n");
   for (ip = *classlist; ip != NULL; ip=ip->next)
      {
      printf("                             %s\n",ip->name);
      }
   }

for (i = 0; ACTIONSEQTEXT[i] != NULL; i++)
   {
   if (strcmp(ACTIONSEQTEXT[i],actiontxt) == 0)
      {
      return (enum aseq) i;
      }
   }

if (strncmp(actiontxt,"module:",7) == 0)
   {
   CheckForModule(actiontxt);
   return plugin;
   }

return(non);
}

/*******************************************************************/

GetHomeInfo()

{ DIR *dirh;
  struct dirent *dirp;
  struct Item *ip;

if (getuid() != 0)                            
   {
   Debug("Not root, so skipping GetHomeInfo()\n");
   return;
   }

if (!MountPathDefined())
   {
   return;
   }


for (ip = VMOUNTLIST; ip != NULL; ip=ip->next)
   {
   if (IsExcluded(ip->classes))
      {
      continue;
      }
   
   if ((dirh = opendir(ip->name)) == NULL)
      {
      sprintf(OUTPUT,"INFO: Host %s seems to have no (additional) local disks except the OS\n",VDEFAULTBINSERVER.name);
      CfLog(cfverbose,OUTPUT,"");
      sprintf(OUTPUT,"      mounted under %s\n\n",ip->name);
      CfLog(cfverbose,OUTPUT,"");
      return;
      }

   for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
      {
      if (!SensibleFile(dirp->d_name,ip->name))
         {
         continue;
         }

      strcpy(VBUFF,ip->name);
      AddSlash(VBUFF);
      strcat(VBUFF,dirp->d_name);

      if (IsHomeDir(VBUFF))
         {
         sprintf(OUTPUT,"Host defines a home directory %s\n",VBUFF);
	 CfLog(cfverbose,OUTPUT,"");
         }
      else
         {
         sprintf(OUTPUT,"Host defines a potential mount point %s\n",VBUFF);
	 CfLog(cfverbose,OUTPUT,"");
         }

      sprintf(CURRENTPATH,"%s%s",ip->name,dirp->d_name);
      sprintf(CURRENTITEM,"%s:%s",VDEFAULTBINSERVER.name,CURRENTPATH);

      if (! IsItemIn(VMOUNTED,CURRENTITEM))
         {
         if ( MOUNTCHECK && ! RequiredFileSystemOkay(CURRENTPATH) && VERBOSE)
            {
            sprintf(OUTPUT,"Found a mountpoint %s but there was\n",CURRENTPATH);
	    CfLog(cfinform,OUTPUT,"");
            CfLog(cfinform,"nothing mounted on it.\n\n","");
            }
         }
      }
   closedir(dirh);
   }
}

/*******************************************************************/

GetMountInfo ()  /* This is, in fact, the most portable way to read the mount info! */
                 /* Depressing, isn't it? */
{ FILE *pp;
  struct Item *mp;
  char buf1[bufsize],buf2[bufsize],buf3[bufsize];
  char host[maxvarsize], mounton[bufsize];
  int i, TimeOut();

if (!GetLock(ASUniqueName("mountinfo"),"",0,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
   {
   return;
   }

Banner("Building list of currently mounted filesystems");

  /* sscanf(VMOUNTCOMM[VSYSTEMHARDCLASS],"%s",buf1); */
  /* Old BSD scanf crashes here! Why!? workaround: */

for (i=0; VMOUNTCOMM[VSYSTEMHARDCLASS][i] != ' '; i++)
   {
   buf1[i] =  VMOUNTCOMM[VSYSTEMHARDCLASS][i];
   }

buf1[i] = '\0';

signal(SIGALRM,(void *)TimeOut);
alarm(RPCTIMEOUT);

if ((pp = cfpopen(buf1,"r")) == NULL)
   {
   sprintf(OUTPUT,"%s: Can't open %s\n",VPREFIX,buf1);
   CfLog(cferror,OUTPUT,"popen");
   return;
   }

do
   {
   VBUFF[0] = buf1[0] = buf2[0] = buf3[0] = '\0';

   if (ferror(pp))  /* abortable */
      {
      GOTMOUNTINFO = false;
      CfLog(cferror,"Error getting mount info\n","ferror");
      break;
      }
   
   ReadLine(VBUFF,bufsize,pp);

   if (ferror(pp))  /* abortable */
      {
      GOTMOUNTINFO = false;
      CfLog(cferror,"Error getting mount info\n","ferror");
      break;
      }
   
   sscanf(VBUFF,"%s%s%s",buf1,buf2,buf3);

   if (VBUFF[0] == '\n')
      {
      break;
      }

   if (strstr(VBUFF,"not responding"))
      {
      printf("%s: %s\n",VPREFIX,VBUFF);
      }

   if (strstr(VBUFF,"be root"))
      {
      CfLog(cferror,"Mount access is denied. You must be root.\n","");
      CfLog(cferror,"Use the -n option to run safely.","");
      }

   if (strstr(VBUFF,"retrying") || strstr(VBUFF,"denied") || strstr(VBUFF,"backgrounding"))
      {
      continue;
      }

   if (strstr(VBUFF,"exceeded") || strstr(VBUFF,"busy"))
      {
      continue;
      }

   if (strstr(VBUFF,"RPC"))
      {
      if (! SILENT)
         {
         CfLog(cfinform,"There was an RPC timeout. Aborting mount operations.\n","");
         CfLog(cfinform,"Session failed while trying to talk to remote host\n","");
         sprintf(OUTPUT,"%s\n",VBUFF);
	 CfLog(cfinform,OUTPUT);
         }

      GOTMOUNTINFO = false;
      ReleaseCurrentLock();
      cfpclose(pp);
      return;
      }

   switch (VSYSTEMHARDCLASS)
      {
      case sun4:
      case sun3:
      case ultrx: 
      case irix:
      case irix4:
      case irix64:
      case linuxx:
      case GnU:
      case unix_sv:
      case freebsd:
      case netbsd:
      case openbsd:
      case bsd_i:
      case nextstep:
      case bsd4_3:
      case newsos:
      case aos:
      case osf:
      case crayos:
                    if (buf1[0] == '/')
                       {
                       strcpy(host,VDEFAULTBINSERVER.name);
                       strcpy(mounton,buf3);
                       }
                    else
                       {
                       sscanf(buf1,"%[^:]",host);
                       strcpy(mounton,buf3);
                       }

                    break;
      case solaris:
      case solarisx86:
      case hp10:
      case hp:      
                    if (buf3[0] == '/')
                       {
                       strcpy(host,VDEFAULTBINSERVER.name);
                       strcpy(mounton,buf1);
                       }
                    else
                       {
                       sscanf(buf3,"%[^:]",host);
                       strcpy(mounton,buf1);
                       }

                    break;
      case aix:
                   /* skip header */

                    if (buf1[0] == '/')
                       {
                       strcpy(host,VDEFAULTBINSERVER.name);
                       strcpy(mounton,buf2);
                       }
                    else
                       {
                       strcpy(host,buf1);
                       strcpy(mounton,buf3);
                       }
                    break;

      case unused1:
      case unused2:
      case unused3:
                    break;

      default:
                    printf("cfengine software error: case %d = %s\n",VSYSTEMHARDCLASS,CLASSTEXT[VSYSTEMHARDCLASS]);
                    FatalError("System error in GetMountInfo - no such class!");
      }

   InstallMountedItem(host,mounton);
   }

while (!feof(pp));

alarm(0);
signal(SIGALRM,SIG_DFL);
ReleaseCurrentLock();
pclose(pp);
}

/*******************************************************************/

MakePaths()

{ struct File *ip;
  char *sp;
  struct stat statbuf;

Banner("Checking directories:");

for (ip = VMAKEPATH; ip != NULL; ip=ip->next)
   {
   if (IsExcluded(ip->classes))
      {
      continue;
      }

   ResetOutputRoute(ip->log,ip->inform);

   CURRENTPATH[0] = '\0';
   ExpandVarstring(ip->path,CURRENTPATH,"");

   if (CURRENTPATH[strlen(CURRENTPATH)-1] != '/')
      {
      if (!(CURRENTPATH[strlen(CURRENTPATH)-2] == '/' && CURRENTPATH[strlen(CURRENTPATH)-1] == '.'))
	{
	strcat(CURRENTPATH,"/.");
	}
      }

   MakeDirectoriesFor(CURRENTPATH);

   if (stat(CURRENTPATH,&statbuf) == -1)
      {
      sprintf(OUTPUT,"Cannot stat %s\n",CURRENTPATH);
      CfLog(cferror,OUTPUT,"stat");
      ResetOutputRoute('d','d');
      continue;
      }

   CheckExistingFile(ip->path,ip->plus,ip->minus,ip->action,ip->uid,ip->gid,&statbuf,ip,ip->acl_aliases);

   ResetOutputRoute('d','d');
   }
}

/*******************************************************************/

MakeChildLinks()     /* <binserver> should expand to a best fit filesys */

{ struct Link *lp;
  char *sp, *bp;
  struct Item *ip;
  int matched,varstring;
  struct stat statbuf;
  short saveenforce;
  short savesilent;

if (NOLINKS)
   {
   return;
   }

Banner("Checking multiple childlinks:");

for (lp = VCHLINK; lp != NULL; lp = lp->next)
   {
   if (IsExcluded(lp->classes))
      {
      continue;
      }

   saveenforce = ENFORCELINKS;
   ENFORCELINKS = ENFORCELINKS || lp->force;

   savesilent = SILENT;
   SILENT = SILENT || lp->silent;

   ResetOutputRoute(lp->log,lp->inform);
   
   matched = varstring = false;

   for(ip = VBINSERVERS; ip != NULL && (!matched); ip = ip->next)
      {
      CURRENTPATH[0] = CURRENTITEM[0] = '\0';

      if (strcmp(lp->to,"linkchildren") == 0)           /* linkchildren */
         {
         if (stat(lp->from,&statbuf) == -1)
            {
            sprintf(OUTPUT,"Makechildlinks() can't stat %s\n",lp->from);
	    CfLog(cferror,OUTPUT,"stat");
	    ResetOutputRoute('d','d');
            continue;
            }
         LinkChildren(lp->from,lp->type,&statbuf,NULL,NULL,NULL,lp);
         break;
         }

      varstring = ExpandVarbinserv(lp->to,CURRENTPATH,ip->name);

      if (lp->recurse != 0)
	 {
	 matched = RecursiveLink(lp,lp->from,CURRENTPATH,lp->recurse);
         }
      else if (LinkChildFiles(lp->from,CURRENTPATH,lp->type,lp->inclusions,lp->exclusions,lp->copy,lp->nofile,lp))
         {
         matched = true;
         }
      else if (! varstring)
         {
         sprintf(OUTPUT,"Error while trying to childlink %s -> %s\n",lp->from,CURRENTPATH);
	 CfLog(cferror,OUTPUT,"");
         sprintf(OUTPUT,"The directory %s does not exist. Can't link.\n",CURRENTPATH);
	 CfLog(cferror,OUTPUT,"");	 
         }

      if (! varstring)                       /* don't iterate over binservers if not var */
         {
         break;
         }
      }

   ENFORCELINKS = saveenforce;
   SILENT = savesilent;
   ResetOutputRoute('d','d');

   if (matched == false && ip == NULL)
      {
      sprintf(OUTPUT,"ChildLink didn't find any server to match %s -> %s\n",lp->from,lp->to);
      CfLog(cferror,OUTPUT,"");
      }
   }
}

/*******************************************************************/

MakeLinks()     /* <binserver> should expand to a best fit filesys */

{ struct Link *lp;
  char *sp, *bp;
  struct Item *ip;
  int matched,varstring;
  struct stat statbuf;
  short saveenforce;
  short savesilent;
  int LinkFiles(), HardLinkFiles(), AbsoluteLink(), RelativeLink();
  int (*linkfiles)();

if (NOLINKS)
   {
   return;
   }

Banner("Checking links:");

for (lp = VLINK; lp != NULL; lp = lp->next)
   {
   if (IsExcluded(lp->classes))
      {
      continue;
      }

   ResetOutputRoute(lp->log,lp->inform);
   
   switch (lp->type)
      {
      case 's':
                linkfiles = LinkFiles;
                break;
      case 'r':
	        linkfiles = RelativeLink;
		break;
      case 'a':
	        linkfiles = AbsoluteLink;
                break;
      case 'h':
                linkfiles = HardLinkFiles;
                break;
      default:
                printf("%s: internal error, link type was [%c]\n",VPREFIX,lp->type);
                continue;
      }

   saveenforce = ENFORCELINKS;
   ENFORCELINKS = ENFORCELINKS || lp->force;

   savesilent = SILENT;
   SILENT = SILENT || lp->silent;

   matched = varstring = false;

   for( ip = VBINSERVERS; ip != NULL && (!matched); ip = ip->next)
      {
      CURRENTPATH[0] = CURRENTITEM[0] = '\0';

      varstring = ExpandVarbinserv(lp->to,CURRENTPATH,ip->name);

      if ((*linkfiles)(lp->from,CURRENTPATH,lp->inclusions,lp->exclusions,lp->copy,lp->nofile,lp))
         {
         matched = true;
         }
      else if (! varstring)
         {
         sprintf(OUTPUT,"Error while trying to link %s -> %s\n",lp->from,CURRENTPATH);
	 CfLog(cferror,OUTPUT,"");
         sprintf(OUTPUT,"The file %s does not exist. Can't link.\n",CURRENTPATH);
	 CfLog(cferror,OUTPUT,"");
         }

      if (! varstring)                       /* don't iterate over binservers if not var */
         {
         break;
         }
      }

   ENFORCELINKS = saveenforce;
   SILENT = savesilent;

   ResetOutputRoute('d','d');
   
   if (matched == false && ip == NULL)
      {
      sprintf(OUTPUT,"Links didn't find any file to match %s -> %s\n",lp->from,lp->to);
      CfLog(cferror,OUTPUT,"");
      }
   }
}

/*******************************************************************/

MailCheck()

{ char mailserver[bufsize];
  char mailhost[maxvarsize];
  char rmailpath[maxvarsize];
  char lmailpath[maxvarsize];


if (VMAILSERVER[0] == '\0')
   {
   FatalError("Program does not define a mailserver for this host");
   }

Banner("Checking mail spool directory");

if (getuid() != 0)                            
   {
   printf("%s: Only root can alter the mail configuration.\n",VPREFIX);
   return;
   }

sscanf (VMAILSERVER,"%[^:]:%s",mailhost,rmailpath);

if (VMAILSERVER[0] == '\0')
   {
   CfLog(cferror,"\n%s: Host has no defined mailserver!\n","");
   return;
   }

if (strcmp(VDEFAULTBINSERVER.name,mailhost) == 0) /* Is this the mailserver ?*/
   {
   MailCheckMore(rmailpath);
   return;
   }

sprintf(lmailpath,"%s:%s",mailhost,VMAILDIR[VSYSTEMHARDCLASS]);


if (IsItemIn(VMOUNTED,lmailpath))                             /* Remote file system mounted on */
   {                                                          /* local mail dir - correct      */
   Verbose("%s: Mail spool area looks ok\n",VPREFIX);
   return;
   }

strcpy(mailserver,VMAILDIR[VSYSTEMHARDCLASS]);
AddSlash(mailserver);
strcat(mailserver,".");

MakeDirectoriesFor(mailserver);                                  /* Check directory is in place */

if (IsItemIn(VMOUNTED,VMAILSERVER))
   {
   if (!SILENT)
      {
      Verbose("%s: Warning - the mail directory seems to be mounted as on\n",VPREFIX);
      Verbose("%s: the remote mailserver and not on the correct local directory\n",VPREFIX);
      Verbose("%s: Should strictly mount on %s\n",VPREFIX,VMAILDIR[VSYSTEMHARDCLASS]);
      }
   return;
   }

if (MatchStringInFstab("mail"))
   {
   if (!SILENT)
      {
      Verbose("%s: Warning - the mail directory seems to be mounted\n",VPREFIX);
      Verbose("%s: in a funny way. I can find the string <mail> in %s\n",VPREFIX,VFSTAB[VSYSTEMHARDCLASS]);
      Verbose("%s: but nothing is mounted on %s\n\n",VPREFIX,VMAILDIR[VSYSTEMHARDCLASS]);
      }
   return;
   }

printf("\n%s: Trying to mount %s\n",VPREFIX,VMAILSERVER);

if (! DONTDO)
   {
   AddToFstab(mailhost,rmailpath,VMAILDIR[VSYSTEMHARDCLASS],"rw",false);
   }
else
   {
   printf("%s: Need to mount %s:%s on %s\n",VPREFIX,mailhost,rmailpath,mailserver);
   }
}


/*******************************************************************/


MailCheckMore(rmailpath)

char *rmailpath;

{ struct stat statbuf;

Verbose("%s: Maildir is %s on %s\n",VPREFIX,rmailpath,VMAILSERVER); 

if (strncmp(VMAILSERVER,VFQNAME,strlen(VMAILSERVER)) != 0)
   { DIR *dirh;
     struct dirent *dirp;
     struct stat statbuf;

   Verbose("%s: Looking closely at the mail directory...\n",VPREFIX);

   if ((dirh = opendir(rmailpath)) == NULL)
      {
      sprintf(OUTPUT,"Can't open directory %s which checking mail directory",rmailpath);
      CfLog(cferror,OUTPUT,"opendir");
      return false;
      }

   for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
      {
      if (!SensibleFile(dirp->d_name,rmailpath))
         {
         continue;
         }

      strcpy(VBUFF,rmailpath);
      AddSlash(VBUFF);
      strcat(VBUFF,dirp->d_name);

      if (stat(VBUFF,&statbuf) != -1)
	 {
	 if (getpwuid(statbuf.st_uid) == NULL)
	    {
	    if (TrueVar("WarnNonOwnerMail"))
	       {
	       sprintf(OUTPUT,"File %s in mail dir %s is not owned by any user",dirp->d_name,rmailpath);
	       CfLog(cferror,OUTPUT,"");
	       }
	    
	    if (TrueVar("DeleteNonOwnerMail"))
	       {
	       if (DONTDO)
		  {
		  printf("%s: Delete file %s\n",VPREFIX,VBUFF);
		  }
	       else
		  {
		  sprintf(OUTPUT,"Deleting file %s in mail dir %s not owned by any user",dirp->d_name,rmailpath);
		  CfLog(cferror,OUTPUT,"");

		  if (unlink(VBUFF) == -1)
		     {
		     CfLog(cferror,"","unlink");
		     }
		  }
	       }
	    continue;
	    }
	 }

      if (strstr(dirp->d_name,"lock") || strstr(dirp->d_name,".tmp"))
	 {
	 Verbose("Ignoring non mailbox file %s\n",dirp->d_name);
	 continue;
	 }

      if (getpwnam(dirp->d_name) == NULL)
	 {
	 if (TrueVar("WarnNonUserMail"))
	    {
	    sprintf(OUTPUT,"File %s in mail dir %s is not the name of a user",dirp->d_name,rmailpath);
	    CfLog(cferror,OUTPUT,"");
	    }
	 
	 if (TrueVar("DeleteNonUserMail"))
	    {
	    if (DONTDO)
	       {
	       printf("%s: Delete file %s\n",VPREFIX,VBUFF);
	       }
	    else
	       {
	       sprintf(OUTPUT,"Deleting file %s in mail dir %s (not a username)",dirp->d_name,rmailpath);
	       CfLog(cferror,OUTPUT,"");
	       
	       if (unlink(VBUFF) == -1)
		  {
		  CfLog(cferror,"","unlink");
		  }	       
	       }
	    }	
	 }
      }
   closedir(dirh);
   Verbose("%s: Done with mail directory\n",VPREFIX);
   } 
}

/*******************************************************************/

MountFileSystems()

{ FILE *pp;
  int TimeOut();

if (! GOTMOUNTINFO || DONTDO)
   {
   return;
   }

if (getuid() != 0)                            
   {
   printf("%s: Only root can mount filesystems.\n",VPREFIX);
   return;
   }

Banner("Mounting filesystems");

signal(SIGALRM,(void *)TimeOut);
alarm(RPCTIMEOUT);

if ((pp = cfpopen(VMOUNTCOMM[VSYSTEMHARDCLASS],"r")) == NULL)
   {
   sprintf(OUTPUT,"Failed to open pipe from %s\n",VMOUNTCOMM[VSYSTEMHARDCLASS]);
   CfLog(cferror,OUTPUT,"popen");
   return;
   }

while (!feof(pp))
   {
   if (ferror(pp))  /* abortable */
      {
      CfLog(cferror,"Error mounting filesystems\n","ferror");
      break;
      }
   
   ReadLine(VBUFF,bufsize,pp);

   if (ferror(pp))  /* abortable */
      {
      CfLog(cferror,"Error mounting filesystems\n","ferror");
      break;
      }

   if (strstr(VBUFF,"already mounted") || strstr(VBUFF,"exceeded") || strstr(VBUFF,"determined"))
      {
      continue;
      }

   if (strstr(VBUFF,"not supported"))
      {
      continue;
      }

   if (strstr(VBUFF,"denied") || strstr(VBUFF,"RPC") || strstr(VBUFF,"root"))
      {
      CfLog(cferror,"There was a mount error, trying to mount one of the filesystems on this host.\n","");
      sprintf(OUTPUT,"%s\n",VBUFF);
      CfLog(cferror,OUTPUT,"");
      GOTMOUNTINFO = false;
      break;
      }

   if (strstr(VBUFF,"trying") && !strstr(VBUFF,"NFS version 2"))
      {
      CfLog(cferror,"Aborted because MountFileSystems() went into a retry loop.\n","");
      GOTMOUNTINFO = false;
      break;
      }
   }

alarm(0);
signal(SIGALRM,SIG_DFL);
cfpclose(pp);
}

/*******************************************************************/

CheckRequired()

{ struct Disk *rp;
  struct Item *ip;
  int matched,varstring,missing = 0;

Banner("Checking required filesystems");

for (rp = VREQUIRED; rp != NULL; rp = rp->next)
   {
   if (IsExcluded(rp->classes))
      {
      continue;
      }

   ResetOutputRoute(rp->log,rp->inform);
   matched = varstring = false;

   for(ip = VBINSERVERS; ip != NULL && (!matched); ip = ip->next)
      {
      CURRENTPATH[0] = CURRENTITEM[0] = '\0';

      varstring = ExpandVarbinserv(rp->name,CURRENTPATH,ip->name);

      if (RequiredFileSystemOkay(CURRENTPATH))  /* simple or reduced item */
         {
         Verbose("Filesystem %s looks sensible\n",CURRENTPATH);
         matched = true;
         }
      else if (! varstring)
         {
         sprintf(OUTPUT,"The file %s does not exist.\n\n",CURRENTPATH);
	 CfLog(cferror,OUTPUT,"");
	 
         /* Define the class if there was no freespace option. */
         if (rp->freespace == -1)
	    {
	    AddRequiredClasses(rp->define);
	    }
         }

      if (! varstring)                       /* don't iterate over binservers if not var */
         {
         break;
         }

      }

   if ((rp->freespace != -1) && !CheckFreeSpace(CURRENTPATH,rp->freespace))
     {
     Verbose("Free space below %d, defining %s\n",rp->freespace, rp->define);
     AddRequiredClasses(rp->define);
     }

   if (matched == false && ip == NULL)
      {
      printf(" didn't find any file to match the required filesystem %s\n",rp->name);
      missing++;
      }
   }

if (missing)

   { time_t tloc;;

   if ((tloc = time((time_t *)NULL)) == -1)
      {
      printf("Couldn't read system clock\n");
      }
   sprintf(OUTPUT,"\n%s: MESSAGE at %s\n\n",ctime(&tloc));
   CfLog(cferror,OUTPUT,"");
   sprintf(OUTPUT,"There are %d required file(system)s missing on host <%s>\n",missing,VDEFAULTBINSERVER.name);
   CfLog(cferror,OUTPUT,"");   
   CfLog(cferror,"even after all mounts have been attempted.\n","");
   CfLog(cferror,"This may be caused by a failure to mount a network filesystem (check exports)\n","");
   sprintf(OUTPUT,"or because no valid server was specified in the program %s\n\n",VINPUTFILE);
   CfLog(cferror,OUTPUT,"");

   ResetOutputRoute('d','d');
   }
}

/*******************************************************************/

TidyFiles()

   /* Here we start by scanning for any absolute path wildcards */
   /* After that's finished, we go snooping around the homedirs */

{ char basename[bufsize],pathbuff[bufsize];
  struct TidyPattern *tlp;
  struct Tidy *tp;
  struct Item *ip1,*ip2;
  struct stat statbuf;
  int homesearch = 0;
  int TidyWrapper();
  int RecHomeTidyWrapper();

Banner("Tidying by directory");

for (tp = VTIDY; tp != NULL; tp=tp->next)
   {
   if (strncmp(tp->path,"home",4)==0)
      {
      for (tlp = tp->tidylist; tlp != NULL; tlp=tlp->next)
	 {
	 if (!IsExcluded(tlp->classes))
	    {
	    homesearch = 1;
	    break;
	    }
	 }
      continue;
      }

   for (tlp = tp->tidylist; tlp != NULL; tlp=tlp->next)
      {
      Verbose("Directory %s\n",tp->path);
      strcpy(VBUFF,tp->path);
      AddSlash(VBUFF);
      strcat(VBUFF,tlp->pattern);

      if (stat(VBUFF,&statbuf) != -1)   /* not lstat - could confuse user */
         {
         if (S_ISDIR(statbuf.st_mode))
            {
	    if (! tlp->rmdirs)          /* Are we allowed to rm empty dirs? */
	       {
               Verbose("%s: will not delete directories (matching %s)!\n",VPREFIX,tlp->pattern);
	       Verbose("%s: Applies to %s\n",VPREFIX,VBUFF);
               DeleteTidyList(tp->tidylist);
	       tp->tidylist = NULL;
               continue;
	       }
            }
         }
      }

   basename[0] = '\0';

   ExpandWildCardsAndDo(tp->path,basename,TidyWrapper,tp);
   
   DeleteTidyList(tp->tidylist);
   tp->tidylist = NULL;
   }


Debug2("End PATHTIDY:\n");

if (!homesearch)                           /* If there are "home" wildcards */
   {                                 /* Don't go rummaging around the disks */
   Verbose("No home patterns to search\n");
   return;
   }

if (getuid() != 0)                            
   {
   printf("%s: Only root can delete others' files.\n",VPREFIX);
   return;
   }


CfLog(cfinform,"Tidying home directories","");

if (!MountPathDefined())
   {
   return;
   }

for (ip1 = VHOMEPATLIST; ip1 != NULL; ip1=ip1->next)
   {
   for (ip2 = VMOUNTLIST; ip2 != NULL; ip2=ip2->next)
      {
      if (IsExcluded(ip2->classes))
	 {
	 continue;
	 }
      pathbuff[0]='\0';
      basename[0]='\0';
      strcpy(pathbuff,ip2->name);
      AddSlash(pathbuff);
      strcat(pathbuff,ip1->name);

      ExpandWildCardsAndDo(pathbuff,basename,RecHomeTidyWrapper,NULL);
      }
   }

CfLog(cfinform,"Done with home directories","");
}

/*******************************************************************/

Scripts()

{ struct ShellComm *ptr;
  char line[bufsize];
  char comm[20], *sp;
  char execstr[bufsize];
  int print,myuid,mygid,TimeOut();
  FILE *pp;

Banner("Running shell commands");

for (ptr = VSCRIPT; ptr != NULL; ptr=ptr->next)
   {
   if (IsExcluded(ptr->classes))
      {
      continue;
      }

   ResetOutputRoute(ptr->log,ptr->inform);
   myuid = -1;
   mygid = -1;
   
   if (!GetLock(ASUniqueName("shellcommand"),CanonifyName(ptr->name),VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
      {
      continue;
      }
   
   sprintf(OUTPUT,"Executing script %s...(timeout=%d)\n",ptr->name,ptr->timeout);
   CfLog(cfinform,OUTPUT,"");

   if (ptr->uid != -1)
      {
      myuid = getuid();
      Verbose("Changing effective uid to %d\n",ptr->uid);
      
      if (seteuid(ptr->uid) == -1)
	 {
	 sprintf(OUTPUT,"Couldn't set effective uid to %d\n",ptr->uid);
	 CfLog(cferror,OUTPUT,"seteuid");
	 ReleaseCurrentLock();
	 continue;
	 }
      }

   if (ptr->gid != -1)
      {
      mygid = getgid();
      Verbose("Changing effective gid to %d\n",ptr->gid);      

      if (setegid(ptr->uid) == -1)
	 {
	 sprintf(OUTPUT,"Couldn't set effective gid to %d\n",ptr->uid);
	 CfLog(cferror,OUTPUT,"seteuid");
	 ReleaseCurrentLock();
	 continue;
	 }
      }
   
   if (DONTDO)
      {
      printf("%s: execute script %s\n",VPREFIX,ptr->name);
      }
   else
      {
      for (sp = ptr->name; *sp != ' ' && *sp != '\0'; sp++)
	 {
	 }

      if (sp - 10 >= ptr->name)
	 {
	 sp -= 10;   /* copy 15 most relevant characters of command */
	 }
      else
	 {
	 sp = ptr->name;
	 }

      bzero(comm,20);
      bzero(execstr,bufsize);

      strncpy(comm,sp,15);

      if (ptr->timeout != 0)
	 {
         signal(SIGALRM,(void *)TimeOut);
         alarm(ptr->timeout);
         }

      switch (ptr->useshell)
	 {
	 case 'y' :
	     sprintf(execstr,"%s 2>&1",ptr->name);

	     if ((pp = popen(execstr,"r")) == NULL)
		{
		sprintf(OUTPUT,"Couldn't open pipe to command %s\n",ptr->name);
		CfLog(cferror,OUTPUT,"popen");
		ResetOutputRoute('d','d');
		RevertId(myuid,mygid);
		ReleaseCurrentLock();
		continue;
		}
	     break;
	     
	 default:
	     if ((pp = cfpopen(ptr->name,"r")) == NULL)
		{
		sprintf(OUTPUT,"Couldn't open pipe to command %s\n",ptr->name);
		CfLog(cferror,OUTPUT,"cfpopen");
		ResetOutputRoute('d','d');
		RevertId(myuid,mygid);		
		ReleaseCurrentLock();		
		continue;
		}
	     break;	     
	 }

      while (!feof(pp))
	 {
	 if (ferror(pp))  /* abortable */
	    {
	    sprintf(OUTPUT,"Shell command pipe %s\n",ptr->name);
	    CfLog(cferror,OUTPUT,"ferror");
	    break;
	    }
	 
	 ReadLine(line,bufsize,pp);

	 if (ferror(pp))  /* abortable */
	    {
	    sprintf(OUTPUT,"Shell command pipe %s\n",ptr->name);
	    CfLog(cferror,OUTPUT,"ferror");
	    break;
	    }	 

	 print = false;
	 
	 for (sp = line; *sp != '\0'; sp++)
	    {
	    if (! isspace(*sp))
	       {
	       print = true;
	       break;
	       }
	    }
	 
	 if (print)
	    {
	    printf("%s:%s: %s\n",VPREFIX,comm,line);
	    }
	 }
      
      switch (ptr->useshell)
	 {      
	 case 'y': pclose(pp);
	           break;
	 default:  cfpclose(pp);
	 }
      }

   if (ptr->timeout != 0)
      {
      alarm(0);
      signal(SIGALRM,SIG_DFL);
      }

   sprintf(OUTPUT,"Finished script %s\n",ptr->name);
   CfLog(cfinform,OUTPUT,"");

   ResetOutputRoute('d','d');
   RevertId(myuid,mygid);
   ReleaseCurrentLock();
   }
}

/*******************************************************************/

GetSetuidLog()

{ struct Item *filetop = NULL;
  struct Item *ip;
  FILE *fp;
  char *sp;

if (getuid() != 0)                     /* Ignore this if not root */
   {
   return;
   }

if ((fp = fopen(VSETUIDLOG,"r")) == NULL)
   {
   }
else
   {
   while (!feof(fp))
      {
      ReadLine(VBUFF,bufsize,fp);

      if (strlen(VBUFF) == 0)
         {
         continue;
         }

      if ((ip = (struct Item *)malloc (sizeof(struct Item))) == NULL)
         {
         perror("malloc");
         FatalError("GetSetuidList() couldn't allocate memory #1");
         }

      if ((sp = malloc (strlen(VBUFF)+2)) == NULL)
         {
         perror("malloc");
         FatalError("GetSetuidList() couldn't allocate memory #2");
         }

      if (filetop == NULL)
         {
         VSETUIDLIST = filetop = ip;
         }
      else
         {
         filetop->next = ip;
         }

      Debug2("SETUID-LOG: %s\n",VBUFF);

      strcpy(sp,VBUFF);
      ip->name = sp;
      ip->next = NULL;
      filetop = ip;
      }

   fclose(fp);
   }

}

/*******************************************************************/

CheckFiles()                         /* Check through file systems */

{ struct File *ptr;
  int CheckFileWrapper();
  char buffer[bufsize];
  short savetravlinks = TRAVLINKS;
  short savekilloldlinks = KILLOLDLINKS;

Banner("Checking files");

if (TRAVLINKS && (VERBOSE || DEBUG || D2))
   {
   printf("(Default in switched to purge stale links...)\n");
   }

for (ptr = VFILE; ptr != NULL; ptr=ptr->next)
   {
   if (IsExcluded(ptr->classes))
      {
      continue;
      }

   TRAVLINKS = savetravlinks;

   if (ptr->travlinks == 'T')
      {
      TRAVLINKS = true;
      }
   else if (ptr->travlinks == 'F')
      {
      TRAVLINKS = false;
      }
   else if (ptr->travlinks == 'K')
      {
      KILLOLDLINKS = true;
      }

   ResetOutputRoute(ptr->log,ptr->inform);

   if (strncmp(ptr->path,"home",4) == 0)
      {
      CheckHome(ptr);
      continue;
      }

   Verbose("Checking file(s) in %s\n",ptr->path);

   buffer[0] = '\0';
   ExpandWildCardsAndDo(ptr->path,buffer,CheckFileWrapper,ptr);

   ResetOutputRoute('d','d');
   TRAVLINKS = savetravlinks;
   KILLOLDLINKS = savekilloldlinks;
   }
}

/*******************************************************************/

SaveSetuidLog()

{ FILE *fp;
  struct Item *ip;


if (getuid() != 0)                     /* Ignore this if not root */
   {
   return;
   }

if (! DONTDO)
   {
   if ((fp = fopen(VSETUIDLOG,"w")) == NULL)
      {
      sprintf(OUTPUT,"Can't open %s for writing\n",VSETUIDLOG);
      CfLog(cferror,OUTPUT,"fopen");
      return;
      }

   Verbose("Saving the setuid log in %s\n",VSETUIDLOG);

   for (ip = VSETUIDLIST; ip != NULL; ip=ip->next)
      {
      if (!isspace(*(ip->name)) && strlen(ip->name) != 0)
         {                         
         fprintf(fp,"%s\n",ip->name);
         Debug2("SAVE-SETUID-LOG: %s\n",ip->name);
         }
      }

   fclose(fp);
   chmod(VSETUIDLOG,0600);
   }
}

/*******************************************************************/

DisableFiles()

{ struct Disable *dp;
  struct stat statbuf;

Banner("Disable files");

for (dp = VDISABLELIST; dp != NULL; dp=dp->next)
   {
   if (IsExcluded(dp->classes))
      {
      continue;
      }

   ResetOutputRoute(dp->log,dp->inform);
   
   if (lstat(dp->name,&statbuf) == -1)
      {
      Verbose("Filetype %s, %s is not there - ok\n",dp->type,dp->name);
      continue;
      }

   Verbose("Disable checking %s\n",dp->name);

   if (S_ISDIR(statbuf.st_mode))
      {
      sprintf(OUTPUT,"Warning %s is a directory.\n",dp->name);
      CfLog(cferror,OUTPUT,"");
      CfLog(cferror,"I refuse to rename/delete a directory!\n\n","");
      ResetOutputRoute('d','d');
      continue;
      }

   if (S_ISLNK(statbuf.st_mode))
      {
      if (strcmp(dp->type,"file") == 0)
         {
         Verbose("%s: %s is a link, not disabling\n",VPREFIX,dp->name);
	 ResetOutputRoute('d','d');
         continue;
         }

      if (readlink(dp->name,VBUFF,bufsize) == -1)
         {
         sprintf(OUTPUT,"DisableFiles() can't read link %s\n",dp->name);
	 CfLog(cferror,OUTPUT,"readlink");
	 ResetOutputRoute('d','d');
         continue;
         }

      printf("%s: Deleting link %s -> %s\n",VPREFIX,dp->name,VBUFF);

      if (! DONTDO)
         {
         if (unlink(dp->name) == -1)
            {
            sprintf(OUTPUT,"Error while unlinking %s\n",dp->name);
            CfLog(cferror,OUTPUT,"unlink");
	    ResetOutputRoute('d','d');
            continue;
            }
	 
	 AddMultipleClasses(dp->defines);
         }
      }
   else
      {
      if (! S_ISREG(statbuf.st_mode))
         {
         Verbose("%s: %s is not a plain file - won't disable\n",VPREFIX,dp->name);
	 ResetOutputRoute('d','d');
         continue;
         }

      if (strcmp(dp->type,"link") == 0)
         {
         Verbose("%s: %s is a file, not disabling\n",VPREFIX,dp->name);
	 ResetOutputRoute('d','d');
         continue;
         }

      if (stat(dp->name,&statbuf) == -1)
         {
         CfLog(cferror,"Internal; error in Disable\n","");
	 ResetOutputRoute('d','d');
	 return;
         }

      if (dp->size != cfnosize)
	 {
         switch (dp->comp)
	    {
            case '<':  if (statbuf.st_size < dp->size)
	                  {
		          Verbose("cfengine %s is smaller than %d bytes\n",dp->name,dp->size);
		          break;
	                  }
	               Verbose("Size is okay\n");
		       ResetOutputRoute('d','d');
	               continue;
		    
   	    case '=':  if (statbuf.st_size == dp->size)
	                  {
		          Verbose("cfengine %s is equal to %d bytes\n",dp->name,dp->size);
		          break;
	                  }
	                Verbose("Size is okay\n");
			ResetOutputRoute('d','d');
			continue;
		       
	    default: if (statbuf.st_size > dp->size)
	                  {
		          Verbose("cfengine %s is larger than %d bytes\n",dp->name,dp->size);
		          break;
	                  }
	               Verbose("Size is okay\n");
		       ResetOutputRoute('d','d');
	               continue;
	    }
	 }

      if (dp->rotate == 0)
         {
         strcpy(CURRENTPATH,dp->name);
         strcat(CURRENTPATH,".cfdisabled");

         sprintf(OUTPUT,"Disabling file %s\n",dp->name);
         CfLog(cfinform,OUTPUT,"");
	 
         if (! DONTDO)
            {
	    chmod(dp->name, (mode_t)0400);

	    if (! IsItemIn(VREPOSLIST,CURRENTPATH))
	       {
               if (rename(dp->name,CURRENTPATH) == -1)
                  {
                  sprintf(OUTPUT,"Error occurred while renaming %s\n",dp->name);
                  CfLog(cferror,OUTPUT,"rename");
		  ResetOutputRoute('d','d');
                  continue;
	          }

	       if (Repository(CURRENTPATH))
	          {
	          unlink(CURRENTPATH);
	          }
	       
	       AddMultipleClasses(dp->defines);
	       }
            }
         }
      else if (dp->rotate == CF_TRUNCATE)
	 {
         sprintf(OUTPUT,"Truncating (emptying) %s\n",dp->name);
	 CfLog(cfinform,OUTPUT,"");
	 
	 if (! DONTDO)
	    {
            TruncateFile(dp->name);
	    AddMultipleClasses(dp->defines);
	    }
	 }
      else
	 {
	 sprintf(OUTPUT,"Rotating files %s by %d\n",dp->name,dp->rotate);
	 CfLog(cfinform,OUTPUT,"");

	 if (!DONTDO)
	    {
	    RotateFiles(dp->name,dp->rotate);
	    AddMultipleClasses(dp->defines);
	    }
	 }
      }
   ResetOutputRoute('d','d');
   }
}

/*******************************************************************/

MountHomeBinServers()

{ struct Item *mp;
  char host[maxvarsize];
  char mountdir[bufsize];
  char maketo[bufsize];

if (! GOTMOUNTINFO)
   {
   CfLog(cfinform,"Incomplete mount info due to RPC failure.\n","");
   sprintf(OUTPUT,"%s will not be modified on this pass!\n\n",VPREFIX,VFSTAB[VSYSTEMHARDCLASS]);
   CfLog(cfinform,OUTPUT,"");
   return;
   }

if (getuid() != 0)                            
   {
   printf("%s:Only root can mount filesystems.\n",VPREFIX);
   return;
   }

Banner("Checking home and binservers");

for (mp = VMOUNTABLES; mp != NULL; mp=mp->next)
   {
   sscanf(mp->name,"%[^:]:%s",host,mountdir);

   Debug("Mount: checking %s\n",mp->name);

   strcpy(maketo,mountdir);

   if (maketo[strlen(maketo)-1] == '/')
      {
      strcat(maketo,".");
      }
   else
      {
      strcat(maketo,"/.");
      }

   if (strcmp(host,VDEFAULTBINSERVER.name) == 0) /* A host never mounts itself nfs */
      {
      Debug("Skipping host %s\n",host);
      continue;
      }

   if (IsHomeDir(mountdir))
      {
      if (!IsItemIn(VMOUNTED,mp->name) && IsItemIn(VHOMESERVERS,host))
         {
         MakeDirectoriesFor(maketo);
         AddToFstab(host,mountdir,mountdir,"rw",false);
         }
      else if (IsItemIn(VHOMESERVERS,host))
	 {
	 AddToFstab(host,mountdir,mountdir,"rw",true);
	 }
      }
   else
      {
      if (!IsItemIn(VMOUNTED,mp->name) && IsItemIn(VBINSERVERS,host))
         {
         MakeDirectoriesFor(maketo);
         AddToFstab(host,mountdir,mountdir,"rw",false);
         }
      else if (IsItemIn(VBINSERVERS,host))
         {
	 AddToFstab(host,mountdir,mountdir,"rw",true);
         }
      }
   }
}


/*********************************************************************/

MountMisc()

{ struct MiscMount *mp;
  char host[maxvarsize];
  char mountdir[bufsize];
  char maketo[bufsize];
  char mtpt[bufsize];

if (! GOTMOUNTINFO)
   {
   CfLog(cfinform,"Incomplete mount info due to RPC failure.\n","");
   sprintf(OUTPUT,"%s will not be modified on this pass!\n\n",VPREFIX,VFSTAB[VSYSTEMHARDCLASS]);
   CfLog(cfinform,OUTPUT,"");
   return;
   }
 
if (getuid() != 0)                            
   {
   printf("%s: Only root can mount filesystems.\n",VPREFIX);
   return;
   }

Banner("Checking miscellaneous mountables:");

for (mp = VMISCMOUNT; mp != NULL; mp=mp->next)
   {
   sscanf(mp->from,"%[^:]:%s",host,mountdir);

   strcpy(maketo,mp->onto);

   if (maketo[strlen(maketo)-1] == '/')
      {
      strcat(maketo,".");
      }
   else
      {
      strcat(maketo,"/.");
      }

   if (strcmp(host,VDEFAULTBINSERVER.name) == 0) /* A host never mounts itself nfs */
      {
      continue;
      }

   sprintf(mtpt,"%s:%s",host,mp->onto);

   if (!IsItemIn(VMOUNTED,mtpt))
      {
      MakeDirectoriesFor(maketo);
      AddToFstab(host,mountdir,mp->onto,mp->options,false);
      }
   else
      {
      AddToFstab(host,mountdir,mp->onto,mp->options,true);
      }
   }
}

/*********************************************************************/

Unmount()

{ struct UnMount *ptr;
  char comm[bufsize];
  char fs[bufsize];
  struct Item *filelist, *item, *search;
  struct stat statbuf;
  FILE *pp;

Banner("Checking unmounts\n");

if (getuid() != 0)                            
   {
   printf("%s: Only root can unmount filesystems.\n",VPREFIX);
   return;
   }

filelist = NULL;

if (! LoadItemList(&filelist,VFSTAB[VSYSTEMHARDCLASS]))
   {
   sprintf(OUTPUT,"Couldn't open %s!\n",VFSTAB[VSYSTEMHARDCLASS]);
   CfLog(cferror,OUTPUT,"");
   return;
   }

NUMBEROFEDITS = 0;

for (ptr=VUNMOUNT; ptr != NULL; ptr=ptr->next)
   {
   if (IsExcluded(ptr->classes))
      {
      continue;
      }

   sscanf(ptr->name,"%*[^:]:%s",fs);

   sprintf(OUTPUT,"Unmount filesystem %s on %s\n",fs,ptr->name);
   CfLog(cfverbose,OUTPUT,"");

   if (strcmp(fs,"/") == 0 || strcmp(fs,"/usr") == 0)
      {
      CfLog(cfinform,"Request to unmount / or /usr is refused!\n","");
      continue;
      }

   if (IsItemIn(VMOUNTED,ptr->name) && (! DONTDO))
      {
      sprintf(comm,"%s %s",VUNMOUNTCOMM[VSYSTEMHARDCLASS],fs);

      if ((pp = cfpopen(comm,"r")) == NULL)
         {
         sprintf(OUTPUT,"Failed to open pipe from %s\n",VUNMOUNTCOMM[VSYSTEMHARDCLASS]);
	 CfLog(cferror,OUTPUT,"");
         return;
         }

      ReadLine(VBUFF,bufsize,pp);

      if (strstr(VBUFF,"busy") || strstr(VBUFF,"Busy"))
         {
         sprintf(OUTPUT,"umount warned that the device under %s\n",ptr->name);
	 CfLog(cfinform,OUTPUT,"");
         CfLog(cfinform,"was busy. Cannot unmount that device.\n","");
         }

      cfpclose(pp);
      }

   if (ptr->deletedir == 't')
      {
      if (stat(fs,&statbuf) != -1)
	 {
	 if ( ! S_ISDIR(statbuf.st_mode))
	    {
	    sprintf(OUTPUT,"Warning! %s was not a directory.\n",fs);
	    CfLog(cfinform,OUTPUT,"");
	    CfLog(cfinform,"(Unmount) will not delete this!\n","");
	    KillOldLink(fs);
	    }
	 else if (! DONTDO)
	    {
	    if (rmdir(fs) == -1)
	       {
	       sprintf(OUTPUT,"Unable to remove the directory %s\n",fs);
	       CfLog(cferror,OUTPUT,"rmdir");
	       } 
	    else
	       {
	       sprintf(OUTPUT,"Removing directory %s\n",ptr->name);
	       CfLog(cfinform,OUTPUT,"");
	       }
	    }
	 }
      }
      
   if (ptr->deletefstab == 't')
      {
      if (VSYSTEMHARDCLASS == aix)
	 {
	 strcpy (VBUFF,fs);
	 strcat (VBUFF,":");
	 
	 item = LocateNextItemContaining(filelist,VBUFF);
	 
	 if (item->next == NULL)
	    {
	    sprintf(OUTPUT,"Bad format in %s\n",VFSTAB[aix]);
	    CfLog(cferror,OUTPUT,"");
	    }
	 
	 DeleteItem(&filelist,item->next);
	 
	 while (strstr(item->next->name,"="))
	    {
	    DeleteItem(&filelist,item->next);    /* DeleteItem(NULL) is harmless */
	    }
	 }
      else
	 {
	 Debug("Trying to delete filesystem %s from list\n",ptr->name);
	 strcpy (VBUFF,ptr->name);
	 
	 if (VSYSTEMHARDCLASS == ultrx)   /* ensure name is no just a substring */
	    {
	    strcat (VBUFF,":");
	    DeleteItemContaining(&filelist,VBUFF);
	    }
	 else
	    {
	    strcat (VBUFF," ");
	    DeleteItemContaining(&filelist,VBUFF);
	    VBUFF[strlen(VBUFF)-1] = '\t';
	    DeleteItemContaining(&filelist,VBUFF);
	    }
	 }
      }
   }

if ((! DONTDO) && (NUMBEROFEDITS > 0))
   {
   SaveItemList(filelist,VFSTAB[VSYSTEMHARDCLASS]);
   }

DeleteItemList(filelist);
}

/*********************************************************************/

EditFiles()

{ struct Edit *ptr;
  struct stat statbuf;
  char linkname[bufsize];
  char realname[bufsize];

Banner("Editing files");

for (ptr=VEDITLIST; ptr!=NULL; ptr=ptr->next)
   {
   if (strncmp(ptr->fname,"home",4) == 0)
      {
      DoEditHomeFiles(ptr);
      }
   else
      {
      if (lstat(ptr->fname,&statbuf) != -1)
	 {
	 if (S_ISLNK(statbuf.st_mode))
	    {
	    Verbose("%s: File %s is a link, editing real file instead\n",VPREFIX,ptr->fname);

            bzero(linkname,bufsize);
	    bzero(realname,bufsize);
	    
	    if (readlink(ptr->fname,linkname,bufsize) == -1)
	       {
	       sprintf(OUTPUT,"Cannot read link %s\n",ptr->fname);
	       CfLog(cferror,OUTPUT,"readlink");
	       continue;
	       }

	    if (linkname[0] != '/')
	       {
	       strcpy(realname,ptr->fname);
	       ChopLastNode(realname);
	       AddSlash(realname);
	       }
	    
	    if (BufferOverflow(realname,linkname))
	       {
	       sprintf(OUTPUT,"(culprit %s in editfiles)\n",ptr->fname);
	       CfLog(cferror,OUTPUT,"");
	       continue;
	       }
	    
	    strcat(realname,linkname);

	    DoEditFile(ptr,realname);
	    }
	 else
	    {
	    DoEditFile(ptr,ptr->fname);
	    }
	 }
      else
	 {
	 DoEditFile(ptr,ptr->fname);
	 }
      }
   }

EDITVERBOSE = false;
}

/*******************************************************************/

CheckResolv()

{ struct Item *filebase = NULL, *referencefile = NULL;
  struct Item *ip, *ip2;
  FILE *fp;
  char ch;
  int fd, existed = true;

Verbose("Checking config in %s\n",VRESOLVCONF[VSYSTEMHARDCLASS]);
  
if (VDOMAIN == NULL)
   {
   CfLog(cferror,"Domain name not specified. Can't configure resolver\n");
   return;
   }

if (getuid() != 0)                            
   {
   printf("%s: Only root can configure the resolver.\n",VPREFIX);
   return;
   }

if (! LoadItemList(&referencefile,VRESOLVCONF[VSYSTEMHARDCLASS]))
   {
   sprintf(OUTPUT,"Trying to create %s\n",VRESOLVCONF[VSYSTEMHARDCLASS]);
   CfLog(cfinform,OUTPUT,"");
   existed = false;
   
   if ((fd = creat(VRESOLVCONF[VSYSTEMHARDCLASS],0644)) == -1)
      {
      sprintf(OUTPUT,"Unable to create file %s\n",VRESOLVCONF[VSYSTEMHARDCLASS]);
      CfLog(cferror,OUTPUT,"creat");
      return;
      }
   else
      {
      close(fd);
      }
   }

if (existed)
   {
   LoadItemList(&filebase,VRESOLVCONF[VSYSTEMHARDCLASS]);
   }

for (ip = filebase; ip != NULL; ip=ip->next)
   {
   if (strlen(ip->name) == 0)
      {
      continue;
      }

   ch = *(ip->name+strlen(ip->name)-2);

   if (isspace(ch))
      {
      sprintf(OUTPUT,"Deleted line %s ended with a space in %s.\n",ip->name,VRESOLVCONF[VSYSTEMHARDCLASS]);
      CfLog(cfinform,OUTPUT,"");
      CfLog(cfinform,"The resolver doesn't understand this.\n","");
      DeleteItem(&filebase,ip);
      }
   else if (isspace(*(ip->name)))
      {
      sprintf(OUTPUT,"Deleted line %s started with a space in %s.\n",ip->name,VRESOLVCONF[VSYSTEMHARDCLASS]);
      CfLog(cfinform,OUTPUT,"");
      CfLog(cfinform,"The resolver doesn't understand this.\n");
      DeleteItem(&filebase,ip);
      }
   }

DeleteItemStarting(&filebase,"domain");
EditItemsInResolvConf(VRESOLVE,&filebase);
sprintf(VBUFF,"domain %s",ToLowerStr(VDOMAIN));
PrependItem(&filebase,VBUFF,NULL);

if (DONTDO)
   {
   printf("Check %s for editing\n",VRESOLVCONF[VSYSTEMHARDCLASS]);
   }
else if (!ItemListsEqual(filebase,referencefile))
   {
   SaveItemList(filebase,VRESOLVCONF[VSYSTEMHARDCLASS]);
   chmod(VRESOLVCONF[VSYSTEMHARDCLASS],DEFAULTSYSTEMMODE);
   }
else
   {
   Verbose("cfengine: %s is okay\n",VRESOLVCONF[VSYSTEMHARDCLASS]);
   }

DeleteItemList(filebase);
DeleteItemList(referencefile);
}


/*******************************************************************/

MakeImages()

{ struct Image *ip;
  struct stat statbuf;
  struct servent *server;
  int savesilent;

Banner("Checking file images:");

for (ip = VIMAGE; ip != NULL; ip=ip->next)
   {
   if (IsExcluded(ip->classes))
      {
      continue;
      }

   IMAGEBACKUP = true;

   Verbose("Checking copy from %s:%s to %s\n",ip->server,ip->path,ip->destination);

   savesilent = SILENT;

   if (strcmp(ip->action,"silent") == 0)
      {
      SILENT = true;
      }

   ResetOutputRoute(ip->log,ip->inform);
   
   if ((strcmp(ip->server,"localhost") != 0) && (PORTNUMBER == 0))
      {
      if ((server = getservbyname(CFENGINE_SERVICE,"tcp")) == NULL)
	 {
	 CfLog(cferror,"Remember to register cfengine in /etc/services: cfengine 5308/tcp\n","getservbyname");
	 exit (1);
	 }

      PORTNUMBER = (unsigned short)(server->s_port);
      }
   
   if (!GetLock(ASUniqueName("copy"),CanonifyName(ip->destination),VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
      {
      SILENT = savesilent;
      ResetOutputRoute('d','d');
      continue;
      }

   IMAGEBACKUP = ip->backup;

   if (cfstat(ip->path,&statbuf,ip) == -1)
      {
      sprintf(OUTPUT,"Can't stat %s in copy\n",ip->path);
      CfLog(cferror,OUTPUT,"");
      ReleaseCurrentLock();
      SILENT = savesilent;
      ResetOutputRoute('d','d');
      continue;
      }

   if (strncmp(ip->destination,"home",4) == 0)
      {
      HOMECOPY = true;          /* Don't send home backups to repository */
      CheckHomeImages(ip);
      HOMECOPY = false;
      }
   else
      {
      if (S_ISDIR(statbuf.st_mode))
	 {
	 if (ip->purge == 'y')
	    {
	    Verbose("%s: (Destination purging enabled)\n",VPREFIX);
	    }
         RecursiveImage(ip,ip->path,ip->destination,ip->recurse);
	 }
      else
	 {
	 if (! MakeDirectoriesFor(ip->destination))
            {
	    ReleaseCurrentLock();
            SILENT = savesilent;
	    ResetOutputRoute('d','d');
            return;
            }

	 CheckImage(ip->path,ip->destination,ip);
	 }
      }

   ReleaseCurrentLock();
   SILENT = savesilent;
   ResetOutputRoute('d','d');
   }
}

/*******************************************************************/

ConfigureInterfaces()

{ struct Interface *ifp;

Banner("Checking network interfaces"); 

if (GetLock("netconfig",VIFDEV[VSYSTEMHARDCLASS],VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
   {
   if (strlen(VNETMASK) != 0)
      {
      IfConf(VIFDEV[VSYSTEMHARDCLASS],VNETMASK,VBROADCAST);
      }
   
   SetDefaultRoute();
   ReleaseCurrentLock();   
   }
    
for (ifp = VIFLIST; ifp != NULL; ifp=ifp->next)
   {
   if (!GetLock("netconfig",ifp->ifdev,VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
      {
      continue;
      }
   
   IfConf(ifp->ifdev,ifp->netmask,ifp->broadcast);
   SetDefaultRoute();
   ReleaseCurrentLock();
   }
}

/*******************************************************************/

CheckTimeZone()

{ time_t tloc;
  struct stat statbuf;
  struct Item *ip;
  
if (VTIMEZONE == NULL)
   {
   CfLog(cferror,"Program does not define a timezone","");
   return;
   }

for (ip = VTIMEZONE; ip != NULL; ip=ip->next)
   {
#ifndef AOS
#ifndef SUN4

   tzset();

   if (strncmp(tzname[0],ip->name,3) == 0)
      {
      return;
      }

#else

   if ((tloc = time((time_t *)NULL)) == -1)
      {
      printf("Couldn't read system clock\n\n");
      }
   
   if (strncmp(localtime(&tloc)->tm_zone ,ip->name,3) == 0)
      {
      return;
      }
   
#endif /* SUN4 */
#endif /* AOS  */
   }


CfLog(cferror,"Time zone was not within the list of acceptable values",""); 
}

/*******************************************************************/

CheckProcesses()

{ struct Process *pp;
  struct Item *procdata = NULL;
  char *psopts = VPSOPTS[VSYSTEMHARDCLASS];

Banner("Checking Processes:");

if (!LoadProcessTable(&procdata,psopts))
   {
   CfLog(cferror,"Unable to read the process table\n","");
   return;
   }

for (pp = VPROCLIST; pp != NULL; pp=pp->next)
   {
   if (IsExcluded(pp->classes))
      {
      continue;
      }

   ResetOutputRoute(pp->log,pp->inform);
   
   if (strcmp(pp->expr,"SetOptionString") == 0)
      {
      psopts = pp->restart;
      DeleteItemList(procdata);
      procdata = NULL;
      if (!LoadProcessTable(&procdata,psopts))
	 {
	 CfLog(cferror,"Unable to read the process table\n","");
	 }
      }
   else
      {
      DoProcessCheck(pp,procdata);
      }
   
   ResetOutputRoute('d','d');
   }
}

/*******************************************************************/
/* Level 2                                                         */
/*******************************************************************/

CheckOpts(argc,argv)

char **argv;
int argc;

{ extern char *optarg;
  int optindex = 0;
  int c;

while ((c=getopt_long(argc,argv,"MAbKqkhYHd:vlniIf:pPmcCtsSaeEVD:N:LwxXuU",OPTIONS,&optindex)) != EOF)
  {
  switch ((char) c)
      {
      case 'E': printf("%s: the enforce-links option can only be used in interactive mode.\n",VPREFIX);
                printf("%s: Do you really want to blindly enforce ALL links? (y/n)\n",VPREFIX);
                if (getchar() != 'y')
                   {
                   printf("cfengine: aborting\n");
		   closelog();
                   exit(1);
                   }

                ENFORCELINKS = true;
                break;

      case 'f': strcpy(VINPUTFILE,optarg);
                MINUSF = true;
                break;

      case 'd': 

	        if (optarg == NULL)
		   {
		   break;
		   }
		
                switch (*optarg)
                   {
                   case '1': D1 = true;
                             break;
                   case '2': D2 = true;
                             break;
		   case '3': D3 = true;
		             VERBOSE = true;
		             break;
                   default:  DEBUG = true;
                             break;
                   }
                break;

      case 'M':
		NOMODULES = true;
		break;
		
      case 'K': IGNORELOCK = true;
	        break;

      case 'A': IGNORELOCK = true;
	        break;
		
      case 'D': AddCompoundClass(optarg);
                break;

      case 'N': NegateCompoundClass(optarg,&VNEGHEAP);
                break;

      case 'b': FORCENETCOPY = true;
	        break;

      case 'e': NOEDITS = true;
                break;

      case 'i': IFCONF = false;
                break;

      case 'I': INFORM = true;
	        break;

      case 'v': VERBOSE = true;
                break;

      case 'l': TRAVLINKS = true;
                break;

      case 'n': DONTDO = true;
	        IGNORELOCK = true;
                break;

      case 'p': PARSEONLY = true;
                IGNORELOCK = true;
                break;

      case 'm': NOMOUNTS = true;
                break;

      case 'c': NOFILECHECK = true;
                break;

      case 'C': MOUNTCHECK = true;
                break;

      case 't': NOTIDY = true;
                break;

      case 's': NOSCRIPTS = true;
                break;

      case 'a': PRSYSADM = true;
                IGNORELOCK = true;
                break;

      case 'L': KILLOLDLINKS = true;
                break;

      case 'V': printf("GNU %s\n%s\n",VERSION,COPYRIGHT);
	        printf("This program is covered by the GNU Public License and may be\n");
		printf("copied free of charge.  No warranty is implied.\n\n");
                exit(0);

      case 'h': Syntax();
                exit(0);

      case 'x': NOPRECONFIG = true;
                break;

      case 'w': WARNINGS = false;
                break;

      case 'X': NOLINKS = true;
                break;

      case 'k': NOCOPY = true;
                break;

      case 'S': SILENT = true;
                break;

      case 'u': USEENVIRON = true;
                break;

      case 'U': UNDERSCORE_CLASSES = true;
	        break;

      case 'H': NOHARDCLASSES = true;
	        break;

      case 'P': NOPROCS = true;
	        break;

      case 'q': NOSPLAY = true;
	        break;

      case 'Y': CFPARANOID = true;
	        break;

      default:  Syntax();
                exit(1);

      }
   }
}

/*******************************************************************/

GetResource(var)

char *var;

{ int i;

for (i = 0; VRESOURCES[i] != '\0'; i++)
   {
   if (strcmp(VRESOURCES[i],var)==0)
      {
      return i;
      }
   }

sprintf (VBUFF,"Unknown resource %s in %s",var,VRCFILE);

FatalError(VBUFF);

return 0; /* This to placate insight */
}


/*******************************************************************/

ListDefinedClasses()

{ struct Item *ptr;

printf ("\nDefined Classes = ( ");

for (ptr = VHEAP; ptr != NULL; ptr=ptr->next)
   {
   printf("%s ",ptr->name);
   }

printf (")\n");

printf ("\nNegated Classes = ( ");

for (ptr = VNEGHEAP; ptr != NULL; ptr=ptr->next)
   {
   printf("%s ",ptr->name);
   }

printf (")\n\n");

printf ("Installable classes = ( ");

for (ptr = VALLADDCLASSES; ptr != NULL; ptr=ptr->next)
   {
   printf("%s ",ptr->name);
   }

printf (")\n");

if (VEXCLUDECOPY != NULL)
   {
   printf("Patterns to exclude from copies: = (");
   
   for (ptr = VEXCLUDECOPY; ptr != NULL; ptr=ptr->next)
      {
      printf("%s ",ptr->name);
      }

   printf (")\n");
   }

if (VEXCLUDELINK != NULL)
   {
   printf("Patterns to exclude from links: = (");
   
   for (ptr = VEXCLUDELINK; ptr != NULL; ptr=ptr->next)
      {
      printf("%s ",ptr->name);
      }

   printf (")\n");
   }

if (VCOPYLINKS != NULL)
   {
   printf("Patterns to copy instead of link: = (");
   
   for (ptr = VCOPYLINKS; ptr != NULL; ptr=ptr->next)
      {
      printf("%s ",ptr->name);
      }

   printf (")\n");
   }

if (VCOPYLINKS != NULL)
   {
   printf("Patterns to link instead of copy: = (");
   
   for (ptr = VLINKCOPIES; ptr != NULL; ptr=ptr->next)
      {
      printf("%s ",ptr->name);
      }

   printf (")\n");
   }

}

/*********************************************************************/

ListDefinedInterfaces()

{ struct Interface *ifp;

 printf ("\nDEFINED INTERFACES\n\n");
 
for (ifp = VIFLIST; ifp !=NULL; ifp=ifp->next)
   {
   printf("Interface %s, netmask=%s, broadcast=%s\n",ifp->ifdev,ifp->netmask,ifp->broadcast);
   }
}

/*********************************************************************/

ListDefinedHomePatterns()

{ struct Item *ptr;


printf ("\nDefined wildcards to match home directories = ( ");

for (ptr = VHOMEPATLIST; ptr != NULL; ptr=ptr->next)
   {
   printf("%s ",ptr->name);
   }

printf (")\n");
}

/*********************************************************************/

ListDefinedBinservers()

{ struct Item *ptr;

printf ("\nDefined Binservers = ( ");

for (ptr = VBINSERVERS; ptr != NULL; ptr=ptr->next)
   {
   printf("%s ",ptr->name);
   }

printf (")\n");
}

/*********************************************************************/

ListDefinedLinks()

{ struct Link *ptr;
  struct Item *ip;
  
printf ("\nDEFINED LINKS\n\n");

for (ptr = VLINK; ptr != NULL; ptr=ptr->next)
   {
   printf("From %s -> %s force=%d, attr=%d type=%c nofile=%d\n",ptr->from,ptr->to,ptr->force,ptr->silent,ptr->type, ptr->nofile);
   for (ip = ptr->copy; ip != NULL; ip = ip->next)
      {
      printf(" Copy %s\n",ip->name);
      }

   for (ip = ptr->exclusions; ip != NULL; ip = ip->next)
      {
      printf(" Exclude %s\n",ip->name);
      }

   for (ip = ptr->inclusions; ip != NULL; ip = ip->next)
      {
      printf(" Include %s\n",ip->name);
      }

   for (ip = ptr->ignores; ip != NULL; ip = ip->next)
      {
      printf(" Ignore %s\n",ip->name);
      }

   printf(" Define %s\n",ptr->defines);
   }
}

/*********************************************************************/

ListDefinedLinkchs()

{ struct Link *ptr;
  struct Item *ip;

printf ("\nDEFINED CHILD LINKS\n\n");

for (ptr = VCHLINK; ptr != NULL; ptr=ptr->next)
   {
   printf("[%s->%s] force=%d attr=%d, rec=%d\n",ptr->from,ptr->to,
	  ptr->force,ptr->silent,ptr->recurse);
   
   for (ip = ptr->copy; ip != NULL; ip = ip->next)
      {
      printf(" Copy %s\n",ip->name);
      }

   for (ip = ptr->exclusions; ip != NULL; ip = ip->next)
      {
      printf(" Exclude %s\n",ip->name);
      }
   
   for (ip = ptr->inclusions; ip != NULL; ip = ip->next)
      {
      printf(" Include %s\n",ip->name);
      }

   for (ip = ptr->ignores; ip != NULL; ip = ip->next)
      {
      printf(" Ignore %s\n",ip->name);
      }

   printf(" Define %s\n",ptr->defines);
   }
}

/*********************************************************************/

ListDefinedResolvers()

{ struct Item *ptr;

printf ("\nDEFINED NAMESERVERS\n\n");

for (ptr = VRESOLVE; ptr != NULL; ptr=ptr->next)
   {
   printf("%s\n",ptr->name);
   }

printf (")\n");
}

/*********************************************************************/

ListDefinedScripts()

{ struct ShellComm *ptr;

printf ("\nDEFINED SHELLCOMMANDS\n\n");

for (ptr = VSCRIPT; ptr != NULL; ptr=ptr->next)
   {
   printf("%s: timeout=%d,uid=%d,gid=%d\n",ptr->name,ptr->timeout,ptr->uid,ptr->gid);
   }
}

/*********************************************************************/

ListDefinedImages()

{ struct Image *ptr;
  struct UidList *up;
  struct GidList *gp;
  struct Item *iip;
  struct sockaddr_in raddr;
  
printf ("\nDEFINED FILE IMAGES\n\n");

for (ptr = VIMAGE; ptr != NULL; ptr=ptr->next)
   {
   printf("\n%s\n +%o\n -%o\n dest: %s\n action: %s\n",ptr->path,ptr->plus,ptr->minus,
	  ptr->destination,ptr->action);

   printf(" Size %c %d\n",ptr->comp,ptr->size);
  
   if (ptr->recurse == INFINITERECURSE)
      {
      printf(" recurse=inf\n");
      }
   else
      {
      printf(" recurse=%d\n",ptr->recurse);
      }
   
   printf(" uids = ( ");

   for (up = ptr->uid; up != NULL; up=up->next)
      {
      printf("%d ",up->uid);
      }

   printf(")\n gids = ( ");

   for (gp = ptr->gid; gp != NULL; gp=gp->next)
      {
      printf("%d ",gp->gid);
      }

   printf(")\n\n exclude:");

   for (iip = ptr->acl_aliases; iip != NULL; iip=iip->next)
      {
      printf(" ACL object %s\n",iip->name);
      }

   printf("ignore:");
   
   for (iip = ptr->ignores; iip != NULL; iip = iip->next)
      {
      printf(" %s",iip->name);
      }
   
   printf("\n");
   printf(" symlink:");

   for (iip = ptr->symlink; iip != NULL; iip = iip->next)
      {
      printf(" %s",iip->name);
      }
   
   printf("\n include:");
   
   for (iip = ptr->inclusions; iip != NULL; iip = iip->next)
      {
      printf(" %s",iip->name);
      }
   printf("\n");
   
   printf(" classes = %s\n",ptr->classes);

   printf(" method = %c (time/checksum)\n",ptr->type);

   printf(" server = %s\n",ptr->server);

   printf(" purging = %c\n",ptr->purge);

   if (ptr->dns != NULL)
      {
      raddr.sin_addr.s_addr = (ptr->dns)->s_addr;
      printf(" host entry cache = %d = %s\n",ptr->dns,inet_ntoa(raddr.sin_addr));
      }

   printf(" define = %s\n",ptr->defines);
   
   if (! ptr->backup)
      {
      printf(" NOT BACKED UP\n");
      }

   if (ptr->stealth == 't')
      {
      printf(" Stealth copy\n");
      }
   }
}

/*********************************************************************/

ListDefinedTidy()

{ struct Tidy *ptr;
  struct TidyPattern *tp;

printf ("\nDEFINED TIDY MASKS\n\n");

for (ptr = VTIDY; ptr != NULL; ptr=ptr->next)
   {
   printf("  path=[%s]\n",ptr->path);
   
   for(tp = ptr->tidylist; tp != NULL; tp=tp->next)
      {
      printf("    pat=%s, define=%s, %c-age=%d, size=%d, linkdirs=%c, rmdirs=%c, travlinks=%c ",
	     tp->pattern,tp->defines,tp->searchtype,tp->age,tp->size,tp->dirlinks,tp->rmdirs,tp->travlinks);

      if (tp->recurse == INFINITERECURSE)
	 {
	 printf("recurse=inf\n");
	 }
      else
	 {
	 printf("recurse=%d\n",tp->recurse);
	 }
      }
   }
}

/*********************************************************************/

ListDefinedMountables()

{ struct Item *ptr;

printf ("\nDEFINED MOUNTABLES\n\n");

for (ptr = VMOUNTABLES; ptr != NULL; ptr=ptr->next)
   {
   printf("%s\n",ptr->name);
   }
}

/*********************************************************************/

ListMiscMounts()

{ struct MiscMount *ptr;

printf ("\nDEFINED MISC MOUNTABLES\n\n");

for (ptr = VMISCMOUNT; ptr != NULL; ptr=ptr->next)
   {
   printf("%s on %s (%s)\n",ptr->from,ptr->onto,ptr->options);
   }

printf (")\n");
}

/*********************************************************************/

ListDefinedRequired()

{ struct Disk *ptr;

printf ("\nDEFINED REQUIRE\n\n");

for (ptr = VREQUIRED; ptr != NULL; ptr=ptr->next)
   {
   printf("%s, freespace=%d, define=%s\n",ptr->name,ptr->freespace,ptr->define);
   }
}

/*********************************************************************/

ListDefinedHomeservers()

{ struct Item *ptr;

printf ("\nDefined home servers = ( ");

for (ptr = VHOMESERVERS; ptr != NULL; ptr=ptr->next)
   {
   printf("%s ",ptr->name);
   }

printf (")\n");
}

/*********************************************************************/

ListDefinedDisable()

{ struct Disable *ptr;

printf ("\nDEFINED DISABLE\n\n");

for (ptr = VDISABLELIST; ptr != NULL; ptr=ptr->next)
   {
   printf("%s: rotate=%d, type=%s, size%c%d, defines=%s\n",
	  ptr->name,ptr->rotate,ptr->type,ptr->comp,ptr->size,ptr->defines);
   }
}

/*********************************************************************/

ListDefinedMakePaths()

{ struct File *ptr;
  struct UidList *up;
  struct GidList *gp;
  struct Item *ip;
  
printf ("\nDEFINED DIRECTORIES\n\n");

for (ptr = VMAKEPATH; ptr != NULL; ptr=ptr->next)
   {
   printf("%s\n +%o\n -%o\n %s\n",ptr->path,ptr->plus,ptr->minus,FILEACTIONTEXT[ptr->action]);

   if (ptr->recurse == INFINITERECURSE)
      {
      printf(" recurse=inf\n");
      }
   else
      {
      printf(" recurse=%d\n",ptr->recurse);
      }
   
   printf(" uids = ( ");

   for (up = ptr->uid; up != NULL; up=up->next)
      {
      printf("%d ",up->uid);
      }

   printf(")\n gids = ( ");

   for (gp = ptr->gid; gp != NULL; gp=gp->next)
      {
      printf("%d ",gp->gid);
      }
   printf(")\n\n");

   for (ip = ptr->acl_aliases; ip != NULL; ip=ip->next)
      {
      printf(" ACL object %s\n",ip->name);
      }

   printf(" define = %s\n",ptr->defines);
   }
}

/*********************************************************************/

ListDefinedImports()

{ struct Item *ptr;

printf ("\nDEFINED IMPORTS\n\n");

for (ptr = VIMPORT; ptr != NULL; ptr=ptr->next)
   {
   printf("%s\n",ptr->name);
   }
}

/*********************************************************************/

ListDefinedIgnore()

{ struct Item *ptr;

printf ("\nDEFINED IGNORE\n\n");

for (ptr = VIGNORE; ptr != NULL; ptr=ptr->next)
   {
   printf("%s\n",ptr->name);
   }
}

/*********************************************************************/

ListFiles()

{ struct File *ptr;
  struct Item *ip;
  struct UidList *up;
  struct GidList *gp;

printf ("\nDEFINED FILES\n\n");

for (ptr = VFILE; ptr != NULL; ptr=ptr->next)
   {
   printf("\n%s\n +%o\n -%o\n %s\n travelinks=%c\n",
	  ptr->path,ptr->plus,ptr->minus,FILEACTIONTEXT[ptr->action],ptr->travlinks);

   if (ptr->recurse == INFINITERECURSE)
      {
      printf(" recurse=inf\n");
      }
   else
      {
      printf(" recurse=%d\n",ptr->recurse);
      }
   
   printf(" uids = ( ");

   for (up = ptr->uid; up != NULL; up=up->next)
      {
      printf("%d ",up->uid);
      }

   printf(")\n gids = ( ");

   for (gp = ptr->gid; gp != NULL; gp=gp->next)
      {
      printf("%d ",gp->gid);
      }
   printf(")\n\n");


   for (ip = ptr->acl_aliases; ip != NULL; ip=ip->next)
      {
      printf(" ACL object %s\n",ip->name);
      }
	  
   for (ip = ptr->exclusions; ip != NULL; ip = ip->next)
      {
      printf(" Exclude %s\n",ip->name);
      }

   for (ip = ptr->inclusions; ip != NULL; ip = ip->next)
      {
      printf(" Include %s\n",ip->name);
      }

   for (ip = ptr->ignores; ip != NULL; ip = ip->next)
      {
      printf(" Ignore %s\n",ip->name);
      }
   
   printf(" define = %s\n",ptr->defines);
   }
}

/*******************************************************************/

ListActionSequence()

{ struct Item *ptr;

printf("\nAction sequence = (");

for (ptr=VACTIONSEQ; ptr!=NULL; ptr=ptr->next)
   {
   printf("%s ",ptr->name);
   }

printf(")\n");
}

/*******************************************************************/

ListUnmounts()

{ struct UnMount *ptr;

printf("\nDEFINED UNMOUNTS\n\n");

for (ptr=VUNMOUNT; ptr!=NULL; ptr=ptr->next)
   {
   printf("%s (classes=%s) deletedir=%c deletefstab=%c force=%c\n",ptr->name,ptr->classes,ptr->deletedir,ptr->deletefstab,ptr->force);
   }
}

/*******************************************************************/

ListProcesses()

{ struct Process *ptr;
  struct Item *ip; 
  char *sp;

printf("\nDEFINED PROCESSES\n\n");

for (ptr = VPROCLIST; ptr != NULL; ptr=ptr->next)
   {
   if (ptr->restart == NULL)
      {
      sp = "";
      }
   else
      {
      sp = ptr->restart;
      }
   
   printf("%s\n Restart = %s (useshell=%c)\n matches: %c%d\n signal=%s\n action=%c\n define=%s\n\n",
	  ptr->expr,sp,ptr->useshell,ptr->comp,ptr->matches,SIGNALS[ptr->signal],ptr->action,ptr->defines);

   for (ip = ptr->exclusions; ip != NULL; ip = ip->next)
      {
      printf(" Exclude %s\n",ip->name);
      }

   for (ip = ptr->inclusions; ip != NULL; ip = ip->next)
      {
      printf(" Include %s\n",ip->name);
      }
   }

printf(")\n");
}

/*******************************************************************/

ListACLs()

{ struct CFACL *ptr;
  struct CFACE *ep;

printf("\nDEFINED ACCESS CONTROL LISTS\n\n");

for (ptr = VACLLIST; ptr != NULL; ptr=ptr->next)
   {
   printf("%s (type=%d,method=%c)\n",ptr->acl_alias,ptr->type,ptr->method);
   
   for (ep = ptr->aces; ep != NULL; ep=ep->next)
      {
      if (ep->name != NULL)
         {
         printf(" Type = %s, obj=%s, mode=%s (classes=%s)\n",ep->acltype,ep->name,ep->mode,ep->classes);
         }
      }
   printf("\n");
   }

}

/*******************************************************************/

ListFileEdits()

{ struct Edit *ptr;
  struct Edlist *ep;

printf("\nDEFINED FILE EDITS\n\n");

for (ptr=VEDITLIST; ptr!=NULL; ptr=ptr->next)
   {
   printf("%s\n",ptr->fname);
   for (ep = ptr->actions; ep != NULL; ep=ep->next)
      {
      if (ep->data == NULL)
         {
         printf(" %s [nodata]\n",VEDITNAMES[ep->code]);
         }
      else
         {
         printf(" %s [%s]\n",VEDITNAMES[ep->code],ep->data);
         }
      }
   printf("\n");
   }
}

/*******************************************************************/

BuildClassEnvironment()

{ struct Item *ip;

sprintf(ALLCLASSBUFFER,"%s=\0",CFALLCLASSESVAR);

for (ip = VHEAP; ip != NULL; ip=ip->next)
   {
   if (IsDefinedClass(ip->name))
      {
      if (BufferOverflow(ALLCLASSBUFFER,ip->name))
	 {
	 printf("%s: culprit BuildClassEnvironment()\n",VPREFIX);
	 return;
	 }

      strcat(ALLCLASSBUFFER,ip->name);
      strcat(ALLCLASSBUFFER,":");
      }
   }

for (ip = VALLADDCLASSES; ip != NULL; ip=ip->next)
   {
   if (IsDefinedClass(ip->name))
      {
      if (BufferOverflow(ALLCLASSBUFFER,ip->name))
	 {
	 printf("%s: culprit BuildClassEnvironment()\n",VPREFIX);
	 return;
	 }
      
      strcat(ALLCLASSBUFFER,ip->name);
      strcat(ALLCLASSBUFFER,":");
      }
   }

Debug2("---\nENVIRONMENT: %s\n---\n",ALLCLASSBUFFER);

if (USEENVIRON)
   {
   if (putenv(ALLCLASSBUFFER) == -1)
      {
      perror("putenv");
      }
   }
}

/*******************************************************************/

CheckForModule(actiontxt)

char *actiontxt;

{ struct stat statbuf;
  char *sp, line[bufsize], id[maxvarsize], *value;
  FILE *pp;
  int print;

if (NOMODULES)
   {
   return;
   }
  
bzero(VBUFF,bufsize);

if (GetMacroValue("moduledirectory"))
   {
   ExpandVarstring("$(moduledirectory)",VBUFF,NULL);
   }
else
   {
   sprintf(OUTPUT,"Plugin was called in ActionSequence but moduledirectory was not defined\n");
   CfLog(cfinform,OUTPUT,"");
   return;
   }

AddSlash(VBUFF);
strcat(VBUFF,actiontxt);

if (stat(VBUFF,&statbuf) == -1)
   {
   sprintf(OUTPUT,"(Plug-in %s not found)",VBUFF);
   Banner(OUTPUT);
   return;
   }

if ((statbuf.st_uid != 0) && (statbuf.st_uid != getuid()))
   {
   sprintf(OUTPUT,"Module %s was not owned by uid=%d executing cfengine\n",VBUFF,getuid());
   return;
   }

sprintf(OUTPUT,"Plug-in `%s\'",actiontxt);
Banner(OUTPUT);

if (!GetLock(ASUniqueName("actionseq"),CanonifyName(actiontxt),VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
   {
   return;
   }
   
if ((pp = cfpopen(VBUFF,"r")) == NULL)
   {
   sprintf(OUTPUT,"Couldn't open pipe from %s\n",actiontxt);
   CfLog(cferror,OUTPUT,"cfpopen");
   return;
   }

while (!feof(pp))
   {
   if (ferror(pp))  /* abortable */
      {
      sprintf(OUTPUT,"Shell command pipe %s\n",actiontxt);
      CfLog(cferror,OUTPUT,"ferror");
      break;
      }
   
   ReadLine(line,bufsize,pp);

   if (strlen(line) > maxvarsize)
      {
      sprintf(OUTPUT,"Line from module %s is too long to be sensible\n",actiontxt);
      CfLog(cferror,OUTPUT,"");
      break;
      }
   
   if (ferror(pp))  /* abortable */
      {
      sprintf(OUTPUT,"Shell command pipe %s\n",actiontxt);
      CfLog(cferror,OUTPUT,"ferror");
      break;
      }	 
   
   print = false;
   
   for (sp = line; *sp != '\0'; sp++)
      {
      if (! isspace(*sp))
	 {
	 print = true;
	 break;
	 }
      }
   
   switch (*line)
      {
      case '+':
	  Verbose("Activated classes: %s\n",line+1);
	  AddMultipleClasses(line+1);
	  break;
      case '-':
	  Verbose("Deactivated classes: %s\n",line+1);
	  NegateCompoundClass(line+1,&VNEGHEAP);
	  break;

      default:
	  if (print)
	     {
	     printf("%s:%s: %s\n",VPREFIX,actiontxt,line);
	     }
      }
   }

cfpclose(pp);

ReleaseCurrentLock();
}


/*******************************************************************/

RequiredFileSystemOkay (name)

char *name;

{ struct stat statbuf, localstat;
  DIR *dirh;
  struct dirent *dirp;
  int sizeinbytes = 0, filecount = 0;
  char buff[bufsize];

Debug2("Checking required filesystem %s\n",name);

if (stat(name,&statbuf) == -1)
   {
   return(false);
   }

if (S_ISLNK(statbuf.st_mode))
   {
   KillOldLink(name);
   return(true);
   }

if (S_ISDIR(statbuf.st_mode))
   {
   if ((dirh = opendir(name)) == NULL)
      {
      sprintf(OUTPUT,"Can't open directory %s which checking required/disk\n",name);
      CfLog(cferror,OUTPUT,"opendir");
      return false;
      }

   for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
      {
      if (!SensibleFile(dirp->d_name,name))
         {
         continue;
         }

      filecount++;

      strcpy(buff,name);

      if (buff[strlen(buff)] != '/')
         {
         strcat(buff,"/");
         }

      strcat(buff,dirp->d_name);

      if (lstat(buff,&localstat) == -1)
         {
         if (S_ISLNK(localstat.st_mode))
            {
            KillOldLink(buff);
            continue;
            }

         sprintf(OUTPUT,"Can't stat %s in required/disk\n",buff);
	 CfLog(cferror,OUTPUT,"lstat");
         continue;
         }

      sizeinbytes += localstat.st_size;
      }

   closedir(dirh);

   if (sizeinbytes < SENSIBLEFSSIZE)
      {
      sprintf(OUTPUT,"File system %s is suspiciously small! (%d bytes)\n",name,sizeinbytes);
      CfLog(cferror,OUTPUT,"");
      return(true);
      }

   if (filecount < SENSIBLEFILECOUNT)
      {
      sprintf(OUTPUT,"Filesystem %s has only %d files/directories.\n",name,filecount);
      CfLog(cferror,OUTPUT,"");
      return(true);
      }
   }

return(true);
}


/*******************************************************************/

InstallMountedItem(host,mountdir)

char *host, *mountdir;

{
strcpy (VBUFF,host);
strcat (VBUFF,":");
strcat (VBUFF,mountdir);

if (IsItemIn(VMOUNTED,VBUFF))
   {
   if (! SILENT || !WARNINGS)
      {
      sprintf(OUTPUT,"WARNING mount item %s\n",VBUFF);
      CfLog(cferror,OUTPUT,"");
      CfLog(cferror,"is mounted multiple times!\n","");
      }
   }

AppendItem(&VMOUNTED,VBUFF,NULL);
}

/*******************************************************************/


AddToFstab(host,rmountpt,mountpt,mode,ismounted)

char *host, *mountpt, *rmountpt, *mode;
int ismounted;

{ char fstab[bufsize];
  char *opts;
  FILE *fp;

Debug("AddToFstab(%s)\n",mountpt);

opts = VMOUNTOPTS[VSYSTEMHARDCLASS];

switch (VSYSTEMHARDCLASS)
   {
   case osf:
   case bsd4_3:
   case irix:
   case irix4:
   case irix64:
   case sun3:
   case aos:
   case nextstep:
   case newsos:
   case sun4:    sprintf(fstab,"%s:%s \t %s %s\t%s,%s 0 0",host,rmountpt,mountpt,VNFSTYPE,mode,opts);
                 break;

   case crayos:
                 sprintf(fstab,"%s:%s \t %s %s\t%s,%s",host,rmountpt,mountpt,ToUpperStr(VNFSTYPE),mode,opts);
                 break;
   case ultrx:   sprintf(fstab,"%s@%s:%s:%s:0:0:%s:%s",rmountpt,host,mountpt,mode,VNFSTYPE,opts);
                 break;
   case hp10:
   case hp:      sprintf(fstab,"%s:%s %s \t %s \t %s,%s 0 0",host,rmountpt,mountpt,VNFSTYPE,mode,opts);
                 break;
   case aix:     sprintf(fstab,"%s:\n\tdev\t= %s\n\ttype\t= %s\n\tvfs\t= %s\n\tnodename\t= %s\n\tmount\t= true\n\toptions\t= %s,%s\n\taccount\t= false\n\n",mountpt,rmountpt,VNFSTYPE,VNFSTYPE,host,mode,opts);
                 break;
   case GnU:
   case linuxx:  sprintf(fstab,"%s:%s \t %s \t %s \t %s",host,rmountpt,mountpt,VNFSTYPE,opts);
                 break;

   case netbsd:
   case openbsd:
   case bsd_i:
   case freebsd: sprintf(fstab,"%s:%s \t %s \t %s \t %s,%s 0 0",host,rmountpt,mountpt,VNFSTYPE,mode,opts);
                 break;

   case unix_sv:
   case solarisx86:
   case solaris: sprintf(fstab,"%s:%s - %s %s - yes %s,%s",host,rmountpt,mountpt,VNFSTYPE,mode,opts);
                 break;
   case unused1:
   case unused2:
   case unused3:
   default:      FatalError("AddToFstab(): unknown hard class detected!\n");
                 break;
   }

if (MatchStringInFstab(mountpt))
   {
   if (!ismounted && !SILENT && !strstr(mountpt,"cdrom"))
      {
      sprintf(OUTPUT,"Warning the file system %s seems to be in %s\n",mountpt,VFSTAB[VSYSTEMHARDCLASS]);
      CfLog(cfinform,OUTPUT,"");
      sprintf(OUTPUT,"already, but I was not able to mount it.\n");
      CfLog(cfinform,OUTPUT,"");
      sprintf(OUTPUT,"Check the exports file on host %s? Check for file with same name as dir?\n",host);
      CfLog(cfinform,OUTPUT,"");
      }

   return;
   }

if (DONTDO)
   {
   printf("%s: add filesystem to %s\n",VPREFIX,VFSTAB[VSYSTEMHARDCLASS]);
   printf("%s: %s\n",VPREFIX,fstab);
   }
else
   {
   if ((fp = fopen(VFSTAB[VSYSTEMHARDCLASS],"a")) == NULL)
      {
      sprintf(OUTPUT,"Can't open %s for appending\n",VFSTAB[VSYSTEMHARDCLASS]);
      CfLog(cferror,OUTPUT,"fopen");
      ReleaseCurrentLock();
      return;
      }

   sprintf(OUTPUT,"Adding filesystem to %s\n",VFSTAB[VSYSTEMHARDCLASS]);
   CfLog(cfinform,OUTPUT,"");
   sprintf(OUTPUT,"%s\n",fstab);
   CfLog(cfinform,OUTPUT,"");
   fprintf(fp,"%s\n",fstab);
   fclose(fp);
   
   chmod(VFSTAB[VSYSTEMHARDCLASS],DEFAULTSYSTEMMODE);
   }
}


/*******************************************************************/

CheckFreeSpace (file,kilobytes)

char *file;
int kilobytes;

{ struct stat statbuf;
  int free;

if (stat(file,&statbuf) == -1)
   {
   sprintf(OUTPUT,"Couldn't stat %s checking diskspace\n",file);
   CfLog(cferror,OUTPUT,"");
   return true;
   }

if (IsMountedFileSystem(&statbuf,file,1)) /* only do this on server */
   {
   return true;
   }

if (kilobytes < 0)  /* percentage */
   {
   free = GetDiskUsage(file,cfpercent);
   kilobytes = -1 * kilobytes;
   if (free < kilobytes)
      {
      sprintf(OUTPUT,"Free disk space is under %d%% for partition\n",kilobytes);
      CfLog(cfinform,OUTPUT,"");
      sprintf(OUTPUT,"containing %s (%d%% free)\n",file,free);
      CfLog(cfinform,OUTPUT,"");
      return false;
      }
   }
else
   {
   free = GetDiskUsage(file, cfabs);

   if (free < kilobytes)
      {
      sprintf(OUTPUT,"Disk space under %d kB for partition\n",kilobytes);
      CfLog(cfinform,OUTPUT,"");
      sprintf(OUTPUT,"containing %s (%d kB free)\n",file,free);
      CfLog(cfinform,OUTPUT,"");
      return false;
      }
   }

return true;
}

/*******************************************************************/

AddRequiredClasses(str)

char *str;

{ char *sp;
  char currentitem[maxvarsize];

for (sp = str; *sp != '\0'; sp++)
   {
   currentitem[0] = '\0';

   sscanf(sp,"%[^,:.]",currentitem);

   sp += strlen(currentitem);

   AddClassToHeap(currentitem);
   }
}

/*******************************************************************/

CheckHome(ptr)                      /* iterate check over homedirs */

struct File *ptr;

{ int RecFileCheck();
  struct Item *ip1, *ip2;
  char basename[bufsize],pathbuff[bufsize];

Debug("CheckHome(%s)\n",ptr->path);

if (getuid() != 0)                            
   {
   printf("%s: Only root can check others' files.\n",VPREFIX);
   return;
   }

if (!MountPathDefined())
   {
   return;
   }

for (ip1 = VHOMEPATLIST; ip1 != NULL; ip1=ip1->next)
   {
   for (ip2 = VMOUNTLIST; ip2 != NULL; ip2=ip2->next)
      {
      if (IsExcluded(ip2->classes))
	 {
	 continue;
	 }
      pathbuff[0]='\0';
      basename[0]='\0';
      strcpy(pathbuff,ip2->name);
      AddSlash(pathbuff);
      strcat(pathbuff,ip1->name);
      AddSlash(pathbuff);

      if (strncmp(ptr->path,"home/",5) == 0) /* home/subdir */
	 {
	 strcat(pathbuff,"*");
	 AddSlash(pathbuff);

         if (*(ptr->path+4) != '/')
            {
            sprintf(OUTPUT,"Illegal use of home in files: %s\n",ptr->path);
	    CfLog(cferror,OUTPUT,"");
            return;
            }
         else
            {
            strcat(pathbuff,ptr->path+5);
            }

	 ExpandWildCardsAndDo(pathbuff,basename,RecFileCheck,ptr);
	 }
      else
	 {
	 ExpandWildCardsAndDo(pathbuff,basename,RecFileCheck,ptr);
	 }
      }
   }

}

/*******************************************************************/


EditItemsInResolvConf(from,list)

struct Item *from, **list;

{ struct Item *ip;
  char buf[maxvarsize];

if (from == NULL)
   {
   return;
   }
else
   {
   EditItemsInResolvConf(from->next,list);
   if (isdigit(*(from->name)))
      {
      sprintf(buf,"nameserver %s",from->name);
      }
   else
      {
      strcpy(buf,from->name);
      }
   
   DeleteItemStarting(list,buf); /* del+prep = move to head of list */
   PrependItem(list,buf,NULL);
   return;
   }
}

/*******************************************************************/

/* These functions are wrappers for the real functions so that
    we can abstract the task of parsing general wildcard paths
    like /a*b/bla?/xyz from a single function ExpandWildCardsAndDo() */

/*******************************************************************/

TidyWrapper(startpath,vp)

char *startpath;
void *vp;

{ struct Tidy *tp;

tp = (struct Tidy *) vp;

Debug2("TidyWrapper(%s)\n",startpath);

RecursiveTidySpecialArea(startpath,tp,tp->recurse);
}

/*******************************************************************/

RecHomeTidyWrapper(startpath,vp)

char *startpath;
void *vp;

{
Verbose("Tidying %s...\n",startpath);
RecursiveHomeTidy(startpath,1,vp);
}

/*******************************************************************/

CheckFileWrapper(startpath,ptr)

char *startpath;
struct File *ptr;

{  struct stat statbuf;
   mode_t filemode;
   char *lastnode;
   int fd;

if ((strlen(startpath) == 0) || (startpath == NULL))
   {
   return;
   }
 
if (ptr->action == touch && IsWildCard(ptr->path))
   {
   printf("%s: Can't touch a wildcard! (%s)\n",VPREFIX,ptr->path);
   return;
   }

for (lastnode = startpath+strlen(startpath)-1; *lastnode != '/'; lastnode--)
   {
   }

lastnode++;

if (ptr->inclusions != NULL && !IsWildItemIn(ptr->inclusions,lastnode))
   {
   Debug2("cfengine: skipping non-included pattern %s\n",lastnode);

   if (stat(startpath,&statbuf) != -1)
      {
      if (!S_ISDIR(statbuf.st_mode))
	 {
	 return;  /* assure that we recurse into directories */
	 }
      }
   }

if (IsWildItemIn(ptr->exclusions,lastnode))
   {
   Debug2("Skipping excluded pattern file %s\n",lastnode);
   return;
   }
    
if (stat(startpath,&statbuf) == -1)
   {
   Verbose("%s: File/directory <%s> does not exist!\n",VPREFIX,ptr->path);

   if (TouchDirectory(ptr))                /* files ending in /. */
      {
      MakeDirectoriesFor(startpath);
      ptr->action = fixall;
      *(startpath+strlen(ptr->path)-2) = '\0';       /* trunc /. */
      CheckExistingFile(startpath,ptr->plus,ptr->minus,ptr->action,ptr->uid,ptr->gid,&statbuf,ptr,ptr->acl_aliases);
      return;
      }

   filemode = DEFAULTMODE;      /* Decide the mode for filecreation */
   filemode |=   ptr->plus;
   filemode &= ~(ptr->minus);

   switch (ptr->action)
      {
      case create:
      case touch:   if (! DONTDO)
                       {
                       MakeDirectoriesFor(startpath);
		       
                       if ((fd = creat(ptr->path,filemode)) == -1)
			  { 
			  perror("creat");
			  return;
			  }
		       else
			  {
			  close(fd);
			  }

		       CheckExistingFile(startpath,ptr->plus,ptr->minus,fixall,ptr->uid,ptr->gid,&statbuf,ptr,ptr->acl_aliases);
                       }

                    sprintf(OUTPUT,"Creating file %s, mode = %o\n",ptr->path,filemode);
		    CfLog(cfinform,OUTPUT,"");
                    break;

      case linkchildren: 
      case warnall:
      case warnplain:
      case warndirs:
      case fixplain:
      case fixdirs:
      case fixall:  sprintf(OUTPUT,"File/Dir %s did not exist and was marked (%s)\n",ptr->path,FILEACTIONTEXT[ptr->action]);
	            CfLog(cfinform,OUTPUT,"");
                    break;

      default:      FatalError("cfengine: Internal sofware error: Checkfiles(), bad action\n");
      }
   }
else
   {
   if (TouchDirectory(ptr))  /* Don't check just touch */
      {
      return;
      }

   if (ptr->action == create)
      {
      return;
      }

   if (ptr->action == linkchildren)
      {
      LinkChildren(ptr->path,&statbuf,ptr->uid,ptr->gid,NULL);
      return;
      }

   if (S_ISDIR(statbuf.st_mode) && (ptr->recurse != 0))
      {
      if (!IgnoreFile(startpath,ReadLastNode(startpath),ptr->ignores))
	 {
	 Verbose("%s: Skipping ignored root-search directory %s\n",VPREFIX,startpath);
	 CheckExistingFile(startpath,ptr->plus,ptr->minus,ptr->action,ptr->uid,ptr->gid,&statbuf,ptr,ptr->acl_aliases);
	 }
      RecursiveCheck(startpath,ptr->plus,ptr->minus,ptr->action,ptr->uid,ptr->gid,ptr->recurse,0,ptr); 
      }
   else
      {
      CheckExistingFile(startpath,ptr->plus,ptr->minus,ptr->action,ptr->uid,ptr->gid,&statbuf,ptr,ptr->acl_aliases);
      }
   }
}

/*******************************************************************/

RevertId (uid,gid)

uid_t uid;
gid_t gid;

{
if (uid != -1)
   {
   if (seteuid(uid) == -1)
      {
      sprintf(OUTPUT,"Unable to regain privileges of user %d\n",uid);
      CfLog(cferror,OUTPUT,"seteuid");
      FatalError("Aborting cfengine");
      }
   }

if (gid != -1)
   {
   if (setegid(gid) == -1)
      {
      sprintf(OUTPUT,"Unable to regain privileges of group %d\n",gid);
      CfLog(cferror,OUTPUT,"seteuid");
      FatalError("Aborting cfengine");      
      }
   }
}

/*******************************************************************/
/* Level 3                                                         */
/*******************************************************************/

TouchDirectory(ptr)                     /* True if file path in /. */

struct File *ptr;

{ char *sp;

if (ptr->action == touch)
   {
   sp = ptr->path+strlen(ptr->path)-2;

   if (strcmp(sp,"/.") == 0)
      {
      return(true);
      }
   else
      {
      return false;
      }
   }
else
   {
   return false;
   }
}


/*******************************************************************/

RecFileCheck(startpath,ptr)

char *startpath;
struct File *ptr;

{
Verbose("%s: Checking files in %s...\n",VPREFIX,startpath);
RecursiveCheck(startpath,ptr->plus,ptr->minus,ptr->action,ptr->uid,ptr->gid,ptr->recurse,0,ptr);
}

/*******************************************************************/

Syntax()

{ int i;

printf("GNU cfengine: A system configuration tool \n%s\n%s\n",VERSION,COPYRIGHT);
printf("\n");
printf("Options:\n\n");

for (i=0; OPTIONS[i].name != NULL; i++)
   {
   printf("--%-20s    (-%c)\n",OPTIONS[i].name,(char)OPTIONS[i].val);
   }

printf("\nDebug levels: 1=parsing, 2=running, 3=summary\n");

printf("\nBug reports to bug-cfengine@gnu.org (News: gnu.cfengine.bug)\n");
printf("General help to help-cfengine@gnu.org (News: gnu.cfengine.help)\n");
printf("Info & fixes at http://www.iu.hioslo.no/~mark/cfengine.html\n");
}

/*******************************************************************/
/* Toolkit fstab                                                   */
/*******************************************************************/

MatchStringInFstab(str)

char *str;

{ FILE *fp;

if ((fp = fopen(VFSTAB[VSYSTEMHARDCLASS],"r")) == NULL)
   {
   sprintf(OUTPUT,"Can't open %s for reading\n",VPREFIX,VFSTAB[VSYSTEMHARDCLASS]);
   CfLog(cferror,OUTPUT,"fopen");
   return true; /* write nothing */
   }

while (!feof(fp))
   {
   ReadLine(VBUFF,bufsize,fp);

   if (VBUFF[0] == '#')
      {
      continue;
      }

   if (strstr(VBUFF,str))
      {
      return true;
      }
   }

fclose(fp);
return(false);
}



/* EOF */
